<?php
// 履歴データの読み込み
$historyFile = __DIR__ . '/../../data/history.json';
$history = [];
if (file_exists($historyFile)) {
    $history = json_decode(file_get_contents($historyFile), true) ?: [];
}

// 統計情報の計算
$totalArticles = count($history);
$todayArticles = 0;
$weekArticles = 0;
$modelUsage = [];

$today = date('Y-m-d');
$weekAgo = date('Y-m-d', strtotime('-7 days'));

foreach ($history as $item) {
    $itemDate = date('Y-m-d', strtotime($item['created_at']));
    
    if ($itemDate === $today) {
        $todayArticles++;
    }
    
    if ($itemDate >= $weekAgo) {
        $weekArticles++;
    }
    
    // モデル使用統計
    $model = $item['model'] ?? 'unknown';
    if (!isset($modelUsage[$model])) {
        $modelUsage[$model] = 0;
    }
    $modelUsage[$model]++;
}

// 最新3件の履歴（よりコンパクトなダッシュボード表示のため）
$recentHistory = array_slice($history, 0, 3);
?>

<div class="page-header">
    <h1><i class="bi bi-speedometer2"></i> ダッシュボード</h1>
    <p class="text-muted">AI記事生成ツールの統計情報</p>
</div>

<div class="row mb-3 stat-cards-row">
    <div class="col-6 col-md-3 mb-2">
        <div class="card stat-card">
            <div class="card-body">
                <div class="stat-content">
                    <div class="stat-text">
                        <div class="stat-label">全記事数</div>
                        <div class="stat-value"><?php echo number_format($totalArticles); ?></div>
                    </div>
                    <div class="stat-icon bg-primary text-white">
                        <i class="bi bi-file-text"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-6 col-md-3 mb-2">
        <div class="card stat-card">
            <div class="card-body">
                <div class="stat-content">
                    <div class="stat-text">
                        <div class="stat-label">本日の記事</div>
                        <div class="stat-value"><?php echo number_format($todayArticles); ?></div>
                    </div>
                    <div class="stat-icon bg-success text-white">
                        <i class="bi bi-calendar-check"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-6 col-md-3 mb-2">
        <div class="card stat-card">
            <div class="card-body">
                <div class="stat-content">
                    <div class="stat-text">
                        <div class="stat-label">直近7日間</div>
                        <div class="stat-value"><?php echo number_format($weekArticles); ?></div>
                    </div>
                    <div class="stat-icon bg-info text-white">
                        <i class="bi bi-graph-up"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-6 col-md-3 mb-2">
        <div class="card stat-card">
            <div class="card-body">
                <div class="stat-content">
                    <div class="stat-text">
                        <div class="stat-label">使用モデル数</div>
                        <div class="stat-value"><?php echo count($modelUsage); ?></div>
                    </div>
                    <div class="stat-icon bg-warning text-white">
                        <i class="bi bi-cpu"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>



<div class="row">
    <div class="col-12 col-xl-8 mb-3">
        <div class="card">
            <div class="card-body">
                <h5 class="card-title mb-3"><i class="bi bi-clock-history"></i> 最近の生成履歴</h5>
                <?php if (empty($recentHistory)): ?>
                    <p class="text-muted text-center py-3">まだ記事が生成されていません。</p>
                <?php else: ?>
                    <div class="table-responsive" style="max-height: 280px;">
                        <table class="table table-hover mb-0">
                            <thead class="sticky-top bg-white">
                                <tr>
                                    <th>タイトル</th>
                                    <th>モデル</th>
                                    <th>画像</th>
                                    <th>日時</th>
                                    <th>操作</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recentHistory as $item): ?>
                                    <tr>
                                        <td class="text-truncate" style="max-width: 250px;"><?php echo htmlspecialchars($item['title'] ?? '無題'); ?></td>
                                        <td>
                                            <span class="badge bg-secondary">
                                                <?php 
                                                $modelName = 'Unknown';
                                                foreach ($config['models'] as $provider => $models) {
                                                    if (isset($models[$item['model']])) {
                                                        $modelName = $models[$item['model']]['name'];
                                                        break;
                                                    }
                                                }
                                                echo htmlspecialchars($modelName);
                                                ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if (!empty($item['images'])): ?>
                                                <span class="badge bg-success">
                                                    <i class="bi bi-image"></i> <?php echo count($item['images']); ?>
                                                </span>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="small"><?php echo date('m/d H:i', strtotime($item['created_at'])); ?></td>
                                        <td>
                                            <button type="button" class="btn btn-sm btn-outline-primary" onclick="viewArticleFromDashboard('<?php echo htmlspecialchars($item['id']); ?>')">
                                                <i class="bi bi-eye"></i>
                                            </button>
                                            <button type="button" class="btn btn-sm btn-outline-info" onclick="createSnsDirectly('<?php echo htmlspecialchars($item['id']); ?>')">
                                                <i class="bi bi-share"></i>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="col-12 col-xl-4 mb-3">
        <div class="card mb-2">
            <div class="card-body p-3">
                <h6 class="card-title mb-2"><i class="bi bi-pie-chart"></i> モデル使用状況</h6>
                <?php if (empty($modelUsage)): ?>
                    <p class="text-muted text-center py-2 mb-0 small">データがありません。</p>
                <?php else: ?>
                    <div style="max-height: 200px; overflow-y: auto;">
                        <?php 
                        arsort($modelUsage);
                        $topModels = array_slice($modelUsage, 0, 5, true);
                        foreach ($topModels as $model => $count): 
                            $percentage = ($count / $totalArticles) * 100;
                            $modelInfo = null;
                            foreach ($config['models'] as $provider => $models) {
                                if (isset($models[$model])) {
                                    $modelInfo = $models[$model];
                                    break;
                                }
                            }
                        ?>
                            <div class="mb-2">
                                <div class="d-flex justify-content-between mb-1 small">
                                    <span class="text-truncate" style="max-width: 150px;"><?php echo htmlspecialchars($modelInfo['name'] ?? $model); ?></span>
                                    <span class="text-muted"><?php echo $count; ?>回</span>
                                </div>
                                <div class="progress" style="height: 6px;">
                                    <div class="progress-bar bg-primary" role="progressbar" 
                                         style="width: <?php echo $percentage; ?>%"
                                         aria-valuenow="<?php echo $percentage; ?>" 
                                         aria-valuemin="0" aria-valuemax="100"></div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="card mb-2">
            <div class="card-body p-3">
                <h6 class="card-title mb-2"><i class="bi bi-info-circle"></i> システム情報</h6>
                <table class="table table-sm mb-0 small">
                    <tr>
                        <td class="py-1">バージョン</td>
                        <td class="py-1"><?php echo htmlspecialchars($config['version'] ?? '1.0.0'); ?></td>
                    </tr>
                    <tr>
                        <td class="py-1">PHP</td>
                        <td class="py-1"><?php echo phpversion(); ?></td>
                    </tr>
                    <tr>
                        <td class="py-1">タイムゾーン</td>
                        <td class="py-1"><?php echo htmlspecialchars($config['timezone'] ?? 'Asia/Tokyo'); ?></td>
                    </tr>
                    <tr>
                        <td class="py-1">キャッシュ</td>
                        <td class="py-1">
                            <?php if ($config['enable_cache'] ?? false): ?>
                                <span class="badge bg-success">有効</span>
                            <?php else: ?>
                                <span class="badge bg-secondary">無効</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
        
        <!-- 使用のヒント（コンパクト版） -->
        <div class="card">
            <div class="card-body p-3">
                <h6 class="card-title mb-2"><i class="bi bi-lightbulb text-warning"></i> クイックヒント</h6>
                <div class="small">
                    <p class="mb-1"><strong>効果的な記事生成:</strong></p>
                    <ul class="mb-2 ps-3">
                        <li>タイトルは具体的で魅力的に</li>
                        <li>キーワードは3-5個程度が最適</li>
                        <li>Perplexityで最新情報を反映</li>
                    </ul>
                    <p class="mb-1"><strong>推奨モデル:</strong></p>
                    <ul class="mb-0 ps-3">
                        <li><strong>Claude Opus 4</strong>: 最高性能</li>
                        <li><strong>GPT-4o</strong>: 多言語対応</li>
                        <li><strong>Venice AI</strong>: DMM記事専用</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* ダッシュボード固有のレイアウト修正 */
.page-content {
    padding-bottom: 2rem !important;
}

/* 余白の最適化 */
.mb-2 {
    margin-bottom: 0.5rem !important;
}

.mb-3 {
    margin-bottom: 0.75rem !important;
}

.mb-4 {
    margin-bottom: 1rem !important;
}

.card-body {
    padding: 1rem;
}

.card-body.p-3 {
    padding: 0.75rem !important;
}

.page-header {
    margin-bottom: 1rem;
}

/* カード高さの制限 */
.card {
    min-height: auto;
    max-height: fit-content;
}

/* 統計カード専用スタイル */
.stat-cards-row .stat-card {
    height: 80px !important;
    max-height: 80px !important;
    overflow: hidden;
}

.stat-cards-row .stat-card .card-body {
    padding: 0.75rem !important;
    height: 100%;
    display: flex;
    align-items: center;
}

.stat-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    width: 100%;
    height: 100%;
}

.stat-text {
    display: flex;
    flex-direction: column;
    justify-content: center;
}

.stat-label {
    font-size: 0.7rem;
    color: #6c757d;
    text-transform: uppercase;
    font-weight: 600;
    margin-bottom: 0.25rem;
    line-height: 1;
}

.stat-value {
    font-size: 1.5rem;
    font-weight: 700;
    color: #333;
    line-height: 1;
}

.stat-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.1rem;
    flex-shrink: 0;
}

/* テーブルスタイル */
.table {
    font-size: 0.85rem;
}

.table.table-sm {
    font-size: 0.8rem;
}

.table th {
    background-color: #f8f9fa;
    font-weight: 600;
    text-transform: uppercase;
    font-size: 0.7rem;
    color: #6c757d;
    border-bottom: 2px solid #dee2e6;
    padding: 0.5rem;
}

.table td {
    vertical-align: middle;
    padding: 0.5rem;
}

.table.table-sm td {
    padding: 0.25rem 0.5rem;
}

.badge {
    padding: 0.25em 0.5em;
    font-size: 0.7rem;
}

/* プログレスバー */
.progress {
    height: 6px !important;
}

/* カードタイトル */
.card-title {
    font-size: 1rem;
    margin-bottom: 0.75rem;
}

h6.card-title {
    font-size: 0.9rem;
    font-weight: 600;
}

/* リスト調整 */
ul.ps-3 {
    padding-left: 1.2rem !important;
}

/* レスポンシブ調整 */
@media (max-width: 768px) {
    .stat-cards-row .stat-card {
        height: 70px !important;
        max-height: 70px !important;
    }
    
    .stat-cards-row .stat-card .card-body {
        padding: 0.5rem !important;
    }
    
    .stat-icon {
        width: 32px;
        height: 32px;
        font-size: 0.85rem;
    }
    
    .stat-value {
        font-size: 1.25rem;
    }
    
    .stat-label {
        font-size: 0.65rem;
    }
    
    .table {
        font-size: 0.75rem;
    }
}

/* グリッドレイアウトの修正 */
@media (min-width: 1200px) {
    .col-xl-8 {
        flex: 0 0 auto;
        width: 66.66666667%;
    }
    
    .col-xl-4 {
        flex: 0 0 auto;
        width: 33.33333333%;
    }
}

/* スクロールバー */
.table-responsive::-webkit-scrollbar,
div[style*="overflow-y: auto"]::-webkit-scrollbar {
    width: 6px;
}

.table-responsive::-webkit-scrollbar-track,
div[style*="overflow-y: auto"]::-webkit-scrollbar-track {
    background: #f1f1f1;
}

.table-responsive::-webkit-scrollbar-thumb,
div[style*="overflow-y: auto"]::-webkit-scrollbar-thumb {
    background: #888;
    border-radius: 3px;
}

.table-responsive::-webkit-scrollbar-thumb:hover,
div[style*="overflow-y: auto"]::-webkit-scrollbar-thumb:hover {
    background: #555;
}
</style>

<!-- SNS投稿機能用のJavaScript -->
<script src="../assets/js/sns-dashboard.js?v=<?php echo time(); ?>"></script>

