<?php
// 設定ファイルが読み込まれているか確認
if (!isset($config)) {
    // 直接アクセスの場合はリダイレクト
    header('Location: ../index.php?page=dmm');
    exit;
}

// Venice AIモデルのデフォルトリスト（APIが利用できない場合のフォールバック）
$veniceModels = [
    'text' => [
        ['id' => 'llama-3.3-70b', 'name' => 'Llama 3.3 70B', 'description' => '高性能な汎用モデル'],
        ['id' => 'llama-3.1-405b', 'name' => 'Llama 3.1 405B', 'description' => '最大級の高精度モデル'],
        ['id' => 'claude-opus-4', 'name' => 'Claude Opus 4', 'description' => 'Anthropic製の最新高性能モデル'],
        ['id' => 'gpt-4.1', 'name' => 'GPT-4.1', 'description' => 'OpenAI製の最新モデル']
    ],
    'image' => [
        ['id' => 'fluently-XL', 'name' => 'Fluently XL', 'description' => '高品質画像生成モデル'],
        ['id' => 'stable-diffusion-xl', 'name' => 'Stable Diffusion XL', 'description' => '汎用画像生成モデル']
    ]
];

?>

<div class="page-content" style="min-height: 100vh; padding-bottom: 100px;">
    <div class="page-header mb-4">
        <h1><i class="bi bi-shop"></i> DMM商品記事生成</h1>
        <p class="text-muted">DMM/FANZAの商品情報を取得し、Perplexity + Venice AIで記事を自動生成します</p>
    </div>

<!-- APIキー設定の確認 -->
<div id="apiKeyAlert" class="alert alert-warning d-none">
    <i class="bi bi-exclamation-triangle"></i> 
    <span id="apiKeyMessage"></span>
    <a href="?page=settings" class="alert-link">設定画面</a>で設定してください。
</div>

<!-- Step 1: 商品検索 -->
<div class="card mb-4">
    <div class="card-header bg-primary text-white">
        <h5 class="mb-0"><i class="bi bi-search"></i> Step 1: 商品検索</h5>
    </div>
    <div class="card-body">
        <form id="searchForm">
            <div class="row">
                <div class="col-md-6">
                    <label for="searchKeyword" class="form-label">検索キーワード</label>
                    <input type="text" class="form-control" id="searchKeyword" placeholder="商品名、女優名、メーカー名など">
                </div>
                <div class="col-md-3">
                    <label for="searchService" class="form-label">サービス</label>
                    <select class="form-select" id="searchService" onchange="updateFloorOptions()">
                        <option value="com">DMM.com</option>
                        <option value="adult">FANZA（アダルト）</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label for="searchSort" class="form-label">並び順</label>
                    <select class="form-select" id="searchSort">
                        <option value="rank">人気順</option>
                        <option value="price">価格順</option>
                        <option value="date">新着順</option>
                        <option value="review">レビュー評価順</option>
                    </select>
                </div>
            </div>
            <div class="row mt-3">
                <div class="col-md-6">
                    <label for="searchFloor" class="form-label">カテゴリ（フロア）</label>
                    <select class="form-select" id="searchFloor">
                        <!-- 動的に生成される -->
                    </select>
                    <small class="text-muted" id="floorHelpText">カテゴリを選択してください</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">&nbsp;</label>
                    <div>
                        <button type="submit" class="btn btn-primary btn-lg w-100" id="searchBtn">
                            <i class="bi bi-search"></i> 商品を検索
                        </button>
                    </div>
                </div>
            </div>
        </form>
        
        <!-- 検索結果 -->
        <div id="searchResults" class="mt-4" style="display: none;">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h6 class="mb-0">検索結果 <span id="resultCount" class="text-muted"></span></h6>
                <div id="paginationTop"></div>
            </div>
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th width="100">画像</th>
                            <th>商品名</th>
                            <th>価格</th>
                            <th>評価</th>
                            <th>操作</th>
                        </tr>
                    </thead>
                    <tbody id="productList">
                        <!-- 動的に生成 -->
                    </tbody>
                </table>
            </div>
            <div id="paginationBottom" class="mt-3 d-flex justify-content-center"></div>
        </div>
    </div>
</div>

<!-- Step 2: 商品詳細と記事生成設定 -->
<div class="card mb-4" id="productDetailCard" style="display: none; margin-bottom: 100px;">
    <div class="card-header bg-success text-white">
        <h5 class="mb-0"><i class="bi bi-file-text"></i> Step 2: 記事生成設定</h5>
    </div>
    <div class="card-body" style="overflow: visible;">
        <!-- 選択された商品情報 -->
        <div id="selectedProduct" class="mb-4 p-3 bg-light rounded">
            <!-- 動的に生成 -->
        </div>
        
        <!-- 記事生成設定 -->
        <form id="generateForm">
            <!-- プリセット選択 -->
            <div class="mb-3">
                <label for="presetSelect" class="form-label">プリセット選択</label>
                <div class="input-group">
                    <select class="form-select" id="presetSelect" onchange="loadPreset()">
                        <option value="">-- プリセットを選択 --</option>
                        <?php
                        // プリセットファイルの読み込み
                        $presetsFile = __DIR__ . '/../../data/presets.json';
                        $presets = [];
                        if (file_exists($presetsFile)) {
                            $presets = json_decode(file_get_contents($presetsFile), true) ?: [];
                        }
                        foreach ($presets as $key => $preset): 
                            // すべてのプリセットを表示（フィルタリングを削除）
                        ?>
                            <option value="<?php echo htmlspecialchars($key); ?>" 
                                    data-model="<?php echo htmlspecialchars($preset['model'] ?? ''); ?>"
                                    data-temperature="<?php echo htmlspecialchars($preset['temperature'] ?? 0.7); ?>"
                                    data-max-tokens="<?php echo htmlspecialchars($preset['max_tokens'] ?? 2000); ?>">
                                <?php echo htmlspecialchars($preset['name']); ?>
                            </option>
                        <?php 
                        endforeach; 
                        ?>
                    </select>
                    <button type="button" class="btn btn-outline-secondary" onclick="clearPreset()">
                        <i class="bi bi-x-circle"></i> クリア
                    </button>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="veniceModel" class="form-label">AIモデル</label>
                    <select class="form-select" id="veniceModel">
                        <?php foreach ($config['models'] as $provider => $models): ?>
                            <optgroup label="<?php echo ucfirst($provider); ?>">
                                <?php foreach ($models as $key => $model): ?>
                                    <option value="<?php echo htmlspecialchars($key); ?>" 
                                            <?php echo $key === $config['default_model'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($model['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </optgroup>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-6">
                    <label class="form-label">情報収集オプション</label>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="enablePerplexity" checked>
                        <label class="form-check-label" for="enablePerplexity">
                            Perplexityで追加情報を収集する
                        </label>
                    </div>
                </div>
            </div>
            
            <div class="mb-3">
                <label for="systemPrompt" class="form-label">システムプロンプト</label>
                <textarea class="form-control" id="systemPrompt" rows="10" placeholder="Venice AIに与える指示を記述します。プリセットを選択すると自動的に設定されます。"></textarea>
                <small class="text-muted">プリセットを選択すると、プリセットに設定されたシステムプロンプトが自動的に読み込まれます</small>
            </div>
            
            <div class="mb-3">
                <label for="keywords" class="form-label">キーワード</label>
                <input type="text" class="form-control" id="keywords" placeholder="SEO用のキーワード（カンマ区切り）">
                <small class="text-muted">記事に含めたいキーワードをカンマ区切りで入力</small>
            </div>
            
            <div class="mb-3">
                <label for="additionalInfo" class="form-label">追加情報（任意）</label>
                <textarea class="form-control" id="additionalInfo" rows="3" placeholder="記事に含めたい追加情報があれば入力してください"></textarea>
            </div>
            
            <!-- 画像・動画自動挿入設定 -->
            <div class="card mb-3">
                <div class="card-header">
                    <h6><i class="bi bi-images"></i> 画像・動画の自動挿入設定</h6>
                </div>
                <div class="card-body">
                    <div class="form-check mb-2">
                        <input type="checkbox" class="form-check-input" id="autoInsertImages">
                        <label class="form-check-label" for="autoInsertImages">
                            サンプル画像を最初の見出し（h2タグ）後に自動挿入
                        </label>
                        <small class="text-muted d-block ms-4">記事生成時に商品のサンプル画像を自動的に挿入します</small>
                    </div>
                </div>
            </div>
            
            <!-- 画像生成設定 -->
            <div class="card mb-3">
                <div class="card-header">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="enableImages" checked>
                        <label class="form-check-label" for="enableImages">
                            <strong>画像生成を有効にする</strong>
                        </label>
                    </div>
                </div>
                <div class="card-body" id="imageSettings">
                    <div class="row">
                        <div class="col-md-6">
                            <label for="imageModel" class="form-label">画像生成モデル</label>
                            <select class="form-select" id="imageModel">
                                <?php foreach ($config['image_models'] as $key => $model): ?>
                                    <option value="<?php echo htmlspecialchars($key); ?>">
                                        <?php echo htmlspecialchars($model['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label for="imageStyle" class="form-label">画像スタイル</label>
                            <select class="form-select" id="imageStyle">
                                <?php foreach ($config['image_styles'] as $key => $style): ?>
                                    <option value="<?php echo htmlspecialchars($key); ?>">
                                        <?php echo htmlspecialchars($style); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label for="numImages" class="form-label">画像数</label>
                            <input type="number" class="form-control" id="numImages" min="1" max="10" value="3" onchange="updateDMMImagePrompts()">
                        </div>
                    </div>
                    <div class="mt-3">
                        <label class="form-label">画像生成プロンプト</label>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="radio" name="dmmPromptType" id="dmmSinglePrompt" value="single" checked onchange="toggleDMMPromptType()">
                            <label class="form-check-label" for="dmmSinglePrompt">
                                共通プロンプト（全画像で同じプロンプトを使用）
                            </label>
                        </div>
                        <div class="form-check mb-3">
                            <input class="form-check-input" type="radio" name="dmmPromptType" id="dmmMultiplePrompts" value="multiple" onchange="toggleDMMPromptType()">
                            <label class="form-check-label" for="dmmMultiplePrompts">
                                個別プロンプト（各画像で異なるプロンプトを指定）
                            </label>
                        </div>
                        
                        <div id="dmmSinglePromptArea">
                            <textarea class="form-control" id="imagePrompt" rows="2" placeholder="画像生成の基本プロンプト（オプション）"></textarea>
                            <small class="text-muted">空欄の場合は記事内容に基づいて自動生成されます</small>
                        </div>
                        
                        <div id="dmmMultiplePromptsArea" style="display: none;">
                            <div id="dmmImagePromptsList">
                                <!-- 動的に生成 -->
                            </div>
                            <small class="text-muted">各画像に個別のプロンプトを設定できます。空欄の場合は自動生成されます。</small>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="form-check mb-3">
                <input class="form-check-input" type="checkbox" id="saveHistory" checked>
                <label class="form-check-label" for="saveHistory">
                    履歴に保存
                </label>
            </div>
            
            <button type="submit" class="btn btn-success btn-lg" id="generateBtn">
                <i class="bi bi-magic"></i> 記事を生成
            </button>
        </form>
    </div>
</div>

<!-- 結果表示モーダル -->
<div class="modal fade" id="resultModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="true">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">生成結果</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" style="max-height: calc(100vh - 200px); overflow-y: auto;">
                <div id="resultContent"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="copyDMMArticle()">
                    <i class="bi bi-clipboard"></i> コピー
                </button>
                <button type="button" class="btn btn-primary" onclick="downloadDMMArticle()">
                    <i class="bi bi-download"></i> ダウンロード
                </button>
                <button type="button" class="btn btn-success" onclick="saveDMMPreset()">
                    <i class="bi bi-bookmark"></i> プリセット保存
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// グローバル変数
let selectedProduct = null;
let generatedContent = null;
let currentSearchParams = null;
let currentPage = 1;
let totalCount = 0;
const itemsPerPage = 20;
let veniceModels = <?php echo json_encode($veniceModels); ?>;
let presets = <?php echo json_encode($presets); ?>;

// LocalStorageから新しいプリセットを確認
const newPresetKey = localStorage.getItem('newPresetKey');
const newPresetData = localStorage.getItem('newPresetData');
if (newPresetKey && newPresetData) {
    try {
        const newPreset = JSON.parse(newPresetData);
        presets[newPresetKey] = newPreset;
        
        // プリセット選択リストに追加（すでにDOMロード済みの場合）
        if (document.readyState !== 'loading') {
            const presetSelect = document.getElementById('presetSelect');
            const option = document.createElement('option');
            option.value = newPresetKey;
            option.textContent = newPreset.name;
            presetSelect.appendChild(option);
        } else {
            // DOMロード前の場合は待機
            document.addEventListener('DOMContentLoaded', function() {
                const presetSelect = document.getElementById('presetSelect');
                const option = document.createElement('option');
                option.value = newPresetKey;
                option.textContent = newPreset.name;
                presetSelect.appendChild(option);
            });
        }
        
        // LocalStorageをクリア
        localStorage.removeItem('newPresetKey');
        localStorage.removeItem('newPresetData');
    } catch (e) {
        console.error('新しいプリセットの読み込みエラー:', e);
    }
}

// ページ読み込み時の処理
document.addEventListener('DOMContentLoaded', function() {
    checkAPIKeys(); // APIキーチェックを再度有効化
    // loadVeniceModels(); // 通常のモデルリストを使用するため不要
    updateFloorOptions();
    
    // 画像設定の表示切替
    document.getElementById('enableImages').addEventListener('change', function() {
        document.getElementById('imageSettings').style.display = this.checked ? 'block' : 'none';
        if (this.checked) {
            updateDMMImagePrompts();
        }
    });
});

// APIキーの確認
function checkAPIKeys() {
    const requiredKeys = [];
    const config = <?php echo json_encode([
        'dmm_api_id' => !empty($config['dmm_api_id']) || !empty($config['dmm']['api_id']),
        'dmm_affiliate_id' => !empty($config['dmm_affiliate_id']) || !empty($config['dmm']['affiliate_id']),
        'has_any_ai_key' => !empty($config['api_keys']['openai']) || 
                           !empty($config['api_keys']['anthropic']) || 
                           !empty($config['api_keys']['claude']) || 
                           !empty($config['api_keys']['google']) || 
                           !empty($config['api_keys']['gemini']) || 
                           !empty($config['api_keys']['venice']) ||
                           !empty($config['api_keys']['perplexity']) ||
                           !empty($config['openai_api_key']) ||
                           !empty($config['anthropic_api_key']) ||
                           !empty($config['google_api_key']) ||
                           !empty($config['venice_api_key']) ||
                           !empty($config['perplexity_api_key'])
    ]); ?>;
    
    if (!config.dmm_api_id || !config.dmm_affiliate_id) {
        requiredKeys.push('DMM API');
    }
    
    // いずれかのAIキーが設定されているかチェック
    if (!config.has_any_ai_key) {
        requiredKeys.push('AI API（OpenAI、Anthropic、Google、Venice、Perplexityのいずれか）');
    }
    
    if (requiredKeys.length > 0) {
        document.getElementById('apiKeyAlert').classList.remove('d-none');
        document.getElementById('apiKeyMessage').textContent = 
            requiredKeys.join('、') + 'のキーが設定されていません。';
    } else {
        // すべてのキーが設定されている場合は警告を非表示
        document.getElementById('apiKeyAlert').classList.add('d-none');
    }
}

// Venice AIモデルの読み込み
async function loadVeniceModels() {
    try {
        const response = await fetch('../api/venice-service.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'get_models'
            })
        });
        
        if (response.ok) {
            const data = await response.json();
            if (data.success && data.text_models && data.text_models.length > 0) {
                veniceModels = {
                    text: data.text_models,
                    image: data.image_models || []
                };
                updateModelSelect();
                console.log('Venice AIモデルを読み込みました:', veniceModels.text.length, 'モデル');
            } else {
                console.log('Venice AIモデルの読み込みに失敗。デフォルトモデルを使用します。');
                // デフォルトモデルは既にPHPで設定されているので何もしない
            }
        } else {
            console.log('Venice AI APIへの接続に失敗。デフォルトモデルを使用します。');
        }
    } catch (error) {
        console.error('Venice AIモデルの読み込みエラー:', error);
        console.log('デフォルトモデルを使用します。');
        // デフォルトモデルは既にPHPで設定されているので何もしない
    }
}

// モデル選択肢の更新
function updateModelSelect() {
    const select = document.getElementById('veniceModel');
    
    // 新しいHTMLを構築
    let html = '<optgroup label="テキストモデル">';
    
    veniceModels.text.forEach(model => {
        html += `<option value="${model.id}" title="${model.description || ''}">
                    ${model.name || model.id}
                 </option>`;
    });
    
    html += '</optgroup>';
    
    // 一度に設定
    select.innerHTML = html;
}

// フロアオプションの更新
function updateFloorOptions() {
    const service = document.getElementById('searchService').value;
    const floorSelect = document.getElementById('searchFloor');
    const floorHelpText = document.getElementById('floorHelpText');
    
    // フロアオプションをクリア
    floorSelect.innerHTML = '';
    
    if (service === 'adult') {
        // FANZAのフロア（オロチの動作確認済みの値を使用）
        const fanzaFloors = [
            { value: 'videoa', text: 'ビデオ' },
            { value: 'videoc', text: '素人' },
            { value: 'anime', text: 'アニメ動画' },
            { value: 'nikkatsu', text: '成人映画' },
            { value: 'playgirl', text: 'プレイガール ch.' },
            { value: 'avstation', text: 'AVステーション' },
            { value: 'dream', text: 'ドリームチャンネル' },
            { value: 'digital_doujin', text: '同人' },
            { value: 'digital_pcgame', text: 'PCゲーム' },
            { value: 'comic', text: 'コミック' },
            { value: 'novel', text: '小説' },
            { value: 'dvd', text: 'DVD' },
            { value: 'goods', text: 'グッズ' },
            { value: 'anime_goods', text: 'アニメグッズ' },
            { value: 'pcgame', text: 'PCゲーム（通販）' },
            { value: 'book', text: '本・雑誌' },
            { value: 'doujin', text: '同人（通販）' },
            { value: 'rental', text: 'レンタル' },
            { value: 'ppr', text: '単品レンタル' }
        ];
        
        fanzaFloors.forEach(floor => {
            const option = document.createElement('option');
            option.value = floor.value;
            option.textContent = floor.text;
            if (floor.value === 'videoa') option.selected = true;
            floorSelect.appendChild(option);
        });
        
        floorSelect.disabled = false;
        floorHelpText.textContent = 'FANZAのカテゴリを選択';
    } else {
        // DMM.comのフロア（オロチの動作確認済みの値を使用）
        const dmmFloors = [
            { value: '', text: '-- 全カテゴリ --' },
            { value: 'akb48', text: 'AKB48' },
            { value: 'ske48', text: 'SKE48' },
            { value: 'nmb48', text: 'NMB48' },
            { value: 'hkt48', text: 'HKT48' },
            { value: 'ngt48', text: 'NGT48' },
            { value: 'rod', text: 'REVIVAL!! ON DEMAND' },
            { value: 'comic', text: 'コミック' },
            { value: 'otherbooks', text: 'ビジネス・実用' },
            { value: 'digital_pcgame_dmm', text: 'PCゲーム' },
            { value: 'digital_pcsoft_dmm', text: 'ソフトウェア' },
            { value: 'dvd_dmm', text: 'DVD・Blu-ray' },
            { value: 'cd_dmm', text: 'CD' },
            { value: 'book_dmm', text: '本・コミック' },
            { value: 'hobby', text: 'ホビー' },
            { value: 'rental_dvd_dmm', text: '月額DVDレンタル' },
            { value: 'rental_cd', text: '月額CDレンタル' },
            { value: 'ppr_dvd_dmm', text: '単品DVDレンタル' },
            { value: 'ppr_cd', text: '単品CDレンタル' },
            { value: 'rental_comic', text: 'コミックレンタル' },
            { value: 'fashion_ladies', text: 'レディースファッション' },
            { value: 'fashion_mens', text: 'メンズファッション' },
            { value: 'rental_iroiro', text: 'いろいろレンタル' }
        ];
        
        dmmFloors.forEach(floor => {
            const option = document.createElement('option');
            option.value = floor.value;
            option.textContent = floor.text;
            floorSelect.appendChild(option);
        });
        
        floorSelect.disabled = false;
        floorHelpText.textContent = 'DMM.comのカテゴリを選択（オプション）';
    }
}

// 商品検索フォーム
document.getElementById('searchForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    
    // 検索パラメータを保存
    currentSearchParams = {
        keyword: document.getElementById('searchKeyword').value,
        service: document.getElementById('searchService').value,
        sort: document.getElementById('searchSort').value,
        floor: document.getElementById('searchFloor').value
    };
    
    // 新しい検索の場合は1ページ目から
    currentPage = 1;
    
    // 検索実行
    await searchProducts(currentPage);
});

// 商品検索実行
async function searchProducts(page = 1) {
    const btn = document.getElementById('searchBtn');
    const originalText = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>検索中...';
    
    try {
        const offset = (page - 1) * itemsPerPage + 1;
        
        const response = await fetch('../api/dmm-service.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'search_products',
                ...currentSearchParams,
                hits: itemsPerPage,
                offset: offset
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            totalCount = result.total_count || 0;
            currentPage = page;
            displaySearchResults(result.products);
            updatePagination();
            document.getElementById('searchResults').style.display = 'block';
            
            // 結果件数を表示
            document.getElementById('resultCount').textContent = 
                `（全${totalCount}件中 ${offset}〜${Math.min(offset + itemsPerPage - 1, totalCount)}件を表示）`;
        } else {
            showToast('検索エラー: ' + result.error, 'danger');
        }
        
    } catch (error) {
        showToast('通信エラー: ' + error.message, 'danger');
    } finally {
        btn.disabled = false;
        btn.innerHTML = originalText;
    }
}

// 検索結果の表示
function displaySearchResults(products) {
    const tbody = document.getElementById('productList');
    tbody.innerHTML = '';
    
    if (products.length === 0) {
        tbody.innerHTML = '<tr><td colspan="5" class="text-center">商品が見つかりませんでした</td></tr>';
        return;
    }
    
    products.forEach(product => {
        const tr = document.createElement('tr');
        tr.innerHTML = `
            <td>
                <img src="${product.image}" alt="${product.title}" class="img-thumbnail" style="max-width: 80px;">
            </td>
            <td>
                <strong>${product.title}</strong>
                <br><small class="text-muted">${product.maker}</small>
            </td>
            <td>¥${product.price}</td>
            <td>
                ${product.review.average > 0 ? 
                    `<span class="text-warning">★${product.review.average}</span> (${product.review.count})` : 
                    '-'
                }
            </td>
            <td>
                <button class="btn btn-sm btn-primary" onclick='selectProduct(${JSON.stringify(product).replace(/'/g, "\\'")})'>
                    選択
                </button>
            </td>
        `;
        tbody.appendChild(tr);
    });
}

// ページネーションの更新
function updatePagination() {
    const totalPages = Math.ceil(totalCount / itemsPerPage);
    
    // ページネーションが必要ない場合は非表示
    if (totalPages <= 1) {
        document.getElementById('paginationTop').innerHTML = '';
        document.getElementById('paginationBottom').innerHTML = '';
        return;
    }
    
    // ページネーションHTML生成
    let paginationHtml = '<nav><ul class="pagination mb-0">';
    
    // 前へボタン
    if (currentPage > 1) {
        paginationHtml += `<li class="page-item">
            <a class="page-link" href="#" onclick="changePage(${currentPage - 1}); return false;">
                <i class="bi bi-chevron-left"></i> 前へ
            </a>
        </li>`;
    } else {
        paginationHtml += `<li class="page-item disabled">
            <span class="page-link"><i class="bi bi-chevron-left"></i> 前へ</span>
        </li>`;
    }
    
    // ページ番号
    const startPage = Math.max(1, currentPage - 2);
    const endPage = Math.min(totalPages, startPage + 4);
    
    if (startPage > 1) {
        paginationHtml += `<li class="page-item">
            <a class="page-link" href="#" onclick="changePage(1); return false;">1</a>
        </li>`;
        if (startPage > 2) {
            paginationHtml += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
        }
    }
    
    for (let i = startPage; i <= endPage; i++) {
        if (i === currentPage) {
            paginationHtml += `<li class="page-item active">
                <span class="page-link">${i}</span>
            </li>`;
        } else {
            paginationHtml += `<li class="page-item">
                <a class="page-link" href="#" onclick="changePage(${i}); return false;">${i}</a>
            </li>`;
        }
    }
    
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            paginationHtml += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
        }
        paginationHtml += `<li class="page-item">
            <a class="page-link" href="#" onclick="changePage(${totalPages}); return false;">${totalPages}</a>
        </li>`;
    }
    
    // 次へボタン
    if (currentPage < totalPages) {
        paginationHtml += `<li class="page-item">
            <a class="page-link" href="#" onclick="changePage(${currentPage + 1}); return false;">
                次へ <i class="bi bi-chevron-right"></i>
            </a>
        </li>`;
    } else {
        paginationHtml += `<li class="page-item disabled">
            <span class="page-link">次へ <i class="bi bi-chevron-right"></i></span>
        </li>`;
    }
    
    paginationHtml += '</ul></nav>';
    
    // 上部と下部に設置
    document.getElementById('paginationTop').innerHTML = paginationHtml;
    document.getElementById('paginationBottom').innerHTML = paginationHtml;
}

// ページ変更
async function changePage(page) {
    if (page < 1 || page === currentPage) return;
    
    // スクロール位置を検索結果の上に
    document.getElementById('searchResults').scrollIntoView({ behavior: 'smooth', block: 'start' });
    
    // 検索実行
    await searchProducts(page);
}

// 商品選択
function selectProduct(product) {
    selectedProduct = product;
    console.log('Selected product:', product);
    console.log('Sample images:', product.sample_images);
    
    // サンプル画像のHTML生成（縦並びで大きく表示）
    let sampleImagesHtml = '';
    if (product.sample_images && product.sample_images.length > 0) {
        sampleImagesHtml = '<div class="mt-3"><h6>サンプル画像:</h6><div class="sample-images-container">';
        product.sample_images.slice(0, 6).forEach((img, index) => {
            sampleImagesHtml += `
                <div class="mb-3">
                    <img src="${img}" alt="サンプル画像${index + 1}" 
                         class="img-fluid rounded shadow-sm" 
                         style="cursor: pointer; max-width: 100%; height: auto;" 
                         onclick="window.open('${img}', '_blank')">
                </div>
            `;
        });
        sampleImagesHtml += '</div></div>';
    }
    
    // サンプル動画のHTML生成（アフィリエイトリンク使用）
    let sampleMovieHtml = '';
    if (product.sample_movie && (product.sample_movie.size_560_360 || product.sample_movie.size_720_480)) {
        // 利用可能な最高画質の動画URLを選択
        const movieUrl = product.sample_movie.size_720_480 || product.sample_movie.size_644_414 || 
                        product.sample_movie.size_560_360 || product.sample_movie.size_476_306;
        
        sampleMovieHtml = `
            <div class="mt-3">
                <h6>サンプル動画:</h6>
                <div class="btn-group" role="group">
                    <button class="btn btn-sm btn-primary" onclick="window.open('${movieUrl}', '_blank')">
                        <i class="bi bi-play-circle"></i> サンプル動画を見る
                    </button>
                    <a href="${product.affiliateURL || product.url}" target="_blank" rel="nofollow noopener" 
                       class="btn btn-sm btn-success">
                        <i class="bi bi-cart"></i> 商品ページへ
                    </a>
                </div>
                <p class="small text-muted mt-1">※ サンプル動画はDMM.com/FANZAで視聴できます</p>
            </div>
        `;
    }
    
    // 商品詳細を表示
    document.getElementById('selectedProduct').innerHTML = `
        <div class="row">
            <div class="col-md-2">
                <img src="${product.image}" alt="${product.title}" class="img-fluid rounded">
            </div>
            <div class="col-md-10">
                <h5>${product.title}</h5>
                <p class="mb-1"><strong>価格:</strong> ¥${product.price}</p>
                <p class="mb-1"><strong>メーカー:</strong> ${product.maker}</p>
                ${product.label ? `<p class="mb-1"><strong>レーベル:</strong> ${product.label}</p>` : ''}
                ${product.actress && product.actress.length > 0 ? `<p class="mb-1"><strong>出演:</strong> ${product.actress.join(', ')}</p>` : ''}
                ${product.genre && product.genre.length > 0 ? `<p class="mb-1"><strong>ジャンル:</strong> ${product.genre.join(', ')}</p>` : ''}
                ${product.review && product.review.average > 0 ? `<p class="mb-1"><strong>評価:</strong> <span class="text-warning">★${product.review.average}</span> (${product.review.count}件)</p>` : ''}
                ${product.date ? `<p class="mb-1"><strong>発売日:</strong> ${product.date}</p>` : ''}
                ${product.content_id ? `<p class="mb-1"><strong>品番:</strong> ${product.content_id}</p>` : ''}
            </div>
        </div>
        ${sampleImagesHtml}
        ${sampleMovieHtml}
    `;
    
    document.getElementById('productDetailCard').style.display = 'block';
    
    // モデルリストはconfig.phpから読み込まれているため、Venice AI専用の処理は不要
    
    // スクロールを少し遅延させて確実に表示
    setTimeout(() => {
        const detailCard = document.getElementById('productDetailCard');
        detailCard.scrollIntoView({ behavior: 'smooth', block: 'start' });
        
        // さらに少し上にスクロールして余白を確保
        setTimeout(() => {
            window.scrollBy(0, -100);
        }, 500);
    }, 100);
}

// プリセット読み込み
function loadPreset() {
    const presetKey = document.getElementById('presetSelect').value;
    if (!presetKey) return;
    
    const preset = presets[presetKey];
    if (!preset) return;
    
    // モデルの設定
    if (preset.model) {
        // venice-プレフィックスがある場合は除去、ない場合はそのまま使用
        const modelId = preset.model.startsWith('venice-') ? preset.model.replace('venice-', '') : preset.model;
        const modelSelect = document.getElementById('veniceModel');
        if (modelSelect) {
            // Venice AIモデル以外の場合は、モデル選択をデフォルトに戻す
            const optionExists = Array.from(modelSelect.options).some(option => option.value === modelId);
            if (optionExists) {
                modelSelect.value = modelId;
            } else {
                // モデルが見つからない場合はデフォルトを選択
                modelSelect.selectedIndex = 0;
            }
        }
    }
    
    // システムプロンプトの設定
    if (preset.system_prompt) {
        document.getElementById('systemPrompt').value = preset.system_prompt;
    }
    
    // 画像生成設定
    if (typeof preset.enable_image !== 'undefined') {
        document.getElementById('enableImages').checked = preset.enable_image;
        document.getElementById('imageSettings').style.display = preset.enable_image ? 'block' : 'none';
    }
    
    if (preset.image_model) {
        document.getElementById('imageModel').value = preset.image_model;
    }
    
    if (preset.image_style) {
        document.getElementById('imageStyle').value = preset.image_style;
    }
    
    if (preset.image_prompt) {
        document.getElementById('imagePrompt').value = preset.image_prompt;
    }
    
    // Temperature と Max Tokens (Venice AI設定として保存する必要がある場合)
    // これらは現在のUIには表示されていないが、必要に応じて追加可能
}

// プリセットクリア
function clearPreset() {
    document.getElementById('presetSelect').value = '';
    // フィールドをクリア
    document.getElementById('systemPrompt').value = '';
    document.getElementById('veniceModel').selectedIndex = 0;
    document.getElementById('enableImages').checked = true;
    document.getElementById('imageSettings').style.display = 'block';
}

// DMM画像プロンプトタイプの切り替え
function toggleDMMPromptType() {
    const isMultiple = document.getElementById('dmmMultiplePrompts').checked;
    document.getElementById('dmmSinglePromptArea').style.display = isMultiple ? 'none' : 'block';
    document.getElementById('dmmMultiplePromptsArea').style.display = isMultiple ? 'block' : 'none';
    
    if (isMultiple) {
        updateDMMImagePrompts();
    }
}

// DMM画像プロンプトフィールドの更新
function updateDMMImagePrompts() {
    const numImages = parseInt(document.getElementById('numImages').value);
    const container = document.getElementById('dmmImagePromptsList');
    container.innerHTML = '';
    
    for (let i = 1; i <= numImages; i++) {
        const div = document.createElement('div');
        div.className = 'mb-2';
        div.innerHTML = `
            <label class="form-label">画像 ${i} のプロンプト</label>
            <textarea class="form-control dmm-image-prompt-item" rows="2" placeholder="画像 ${i} の生成プロンプト（空欄可）"></textarea>
        `;
        container.appendChild(div);
    }
}

// 記事生成フォーム
let isGenerating = false; // 重複送信防止フラグ
document.getElementById('generateForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    
    // 既に生成中の場合は処理しない
    if (isGenerating) {
        console.log('既に生成処理中です');
        return;
    }
    
    if (!selectedProduct) {
        showToast('商品を選択してください', 'warning');
        return;
    }
    
    const btn = document.getElementById('generateBtn');
    const originalText = btn.innerHTML;
    
    // 重複送信防止
    isGenerating = true;
    btn.disabled = true;
    btn.style.pointerEvents = 'none'; // クリックイベント自体を無効化
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>生成中...';
    
    try {
        // 画像プロンプトの収集
        let imagePrompts = [];
        const isMultiplePrompts = document.getElementById('dmmMultiplePrompts').checked;
        
        if (isMultiplePrompts) {
            const promptElements = document.querySelectorAll('.dmm-image-prompt-item');
            promptElements.forEach(el => {
                imagePrompts.push(el.value);
            });
        } else {
            // 単一プロンプトの場合は配列として送信
            const singlePrompt = document.getElementById('imagePrompt').value;
            const numImages = parseInt(document.getElementById('numImages').value);
            for (let i = 0; i < numImages; i++) {
                imagePrompts.push(singlePrompt);
            }
        }
        
        const response = await fetch('../api/dmm-service.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'generate_dmm_article',
                product: {
                    ...selectedProduct,
                    sample_images: selectedProduct.sample_images || []  // 明示的にsample_imagesを含める
                },
                keywords: document.getElementById('keywords').value,
                system_prompt: document.getElementById('systemPrompt').value,
                additional_info: document.getElementById('additionalInfo').value,
                venice_model: document.getElementById('veniceModel').value,
                ai_model: document.getElementById('veniceModel').value,  // ai_modelも送信
                enable_perplexity: document.getElementById('enablePerplexity').checked,
                save_history: document.getElementById('saveHistory').checked,
                enable_images: document.getElementById('enableImages').checked,
                image_model: document.getElementById('imageModel').value,
                image_style: document.getElementById('imageStyle').value,
                num_images: parseInt(document.getElementById('numImages').value),
                image_prompts: imagePrompts,  // 複数形に変更
                auto_insert_images: document.getElementById('autoInsertImages').checked  // 自動挿入フラグを追加
            })
        });
        
        // レスポンスのテキストを先に取得
        const responseText = await response.text();
        
        // JSONパースを試みる
        let result;
        try {
            result = JSON.parse(responseText);
        } catch (e) {
            console.error('JSON Parse Error:', e);
            console.error('Response Text:', responseText);
            
            // HTMLレスポンスの場合はエラーメッセージを抽出
            if (responseText.includes('<!DOCTYPE') || responseText.includes('<html')) {
                throw new Error('サーバーエラー: HTMLレスポンスが返されました。ログインが必要な可能性があります。');
            } else if (responseText === '') {
                throw new Error('サーバーエラー: 空のレスポンスが返されました。');
            } else {
                throw new Error('サーバーエラー: 不正なJSONレスポンス - ' + responseText.substring(0, 100));
            }
        }
        
        if (result.success) {
            generatedContent = result.content;
            showResult(result);
        } else {
            showToast('生成エラー: ' + result.error, 'danger');
        }
        
    } catch (error) {
        showToast('通信エラー: ' + error.message, 'danger');
    } finally {
        // 少し遅延してからボタンを有効化（連続クリック防止）
        setTimeout(() => {
            btn.disabled = false;
            btn.style.pointerEvents = 'auto';
            btn.innerHTML = originalText;
            isGenerating = false;
        }, 1000);
    }
});

// 結果表示
function showResult(result) {
    let content = result.content;
    
    // スタイルタグを追加（CTAボタンとモーダルのスタイル）
    const styleTag = `<style>
        .dmm-cta-section {
            margin: 40px 0;
            padding: 30px;
            background-color: #f8f9fa;
            border-radius: 10px;
            text-align: center;
        }
        .dmm-cta-section h3 {
            color: #333;
            margin-bottom: 20px;
        }
        .dmm-cta-section p {
            color: #666;
            margin-bottom: 25px;
        }
        .dmm-cta-section a {
            display: inline-block;
            padding: 15px 40px;
            background-color: #ff6b6b;
            color: white;
            text-decoration: none;
            border-radius: 50px;
            font-size: 18px;
            font-weight: bold;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }
        .dmm-cta-section a:hover {
            background-color: #ff5252;
            transform: translateY(-2px);
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
        .dmm-sample-section {
            margin: 30px 0;
            padding: 20px;
            background-color: #f8f9fa;
            border-radius: 10px;
        }
        .dmm-sample-section h3 {
            color: #333;
            margin-bottom: 15px;
            font-size: 1.3rem;
        }
        .dmm-sample-images {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-bottom: 20px;
        }
        .dmm-sample-images img {
            width: 150px;
            height: auto;
            border-radius: 5px;
            cursor: pointer;
            transition: transform 0.2s;
        }
        .dmm-sample-images img:hover {
            transform: scale(1.05);
        }
    </style>`;
    
    // 商品画像を記事に挿入（最初に必ず挿入）
    let productImageHtml = '';
    if (selectedProduct) {
        // メイン商品画像
        if (selectedProduct.image) {
            productImageHtml = `
                <figure class="text-center my-4">
                    <img src="${selectedProduct.image}" alt="${selectedProduct.title}" class="img-fluid rounded shadow" style="max-width: 600px;">
                    <figcaption class="text-muted small mt-2">${selectedProduct.title}</figcaption>
                </figure>
            `;
        }
        
        // サンプル画像セクション
        if (selectedProduct.sample_images && selectedProduct.sample_images.length > 0) {
            productImageHtml += `
                <div class="dmm-sample-section">
                    <h3>商品サンプル画像</h3>
                    <div class="dmm-sample-images">
            `;
            selectedProduct.sample_images.slice(0, 6).forEach((img, idx) => {
                productImageHtml += `<img src="${img}" alt="サンプル画像${idx + 1}" onclick="window.open('${img}', '_blank')">`;
            });
            productImageHtml += `
                    </div>
                    <div style="text-align: center; margin-top: 20px;">
                        <a href="${selectedProduct.affiliateURL || selectedProduct.url}" target="_blank" rel="nofollow noopener" 
                           style="display: inline-block; padding: 15px 40px; background-color: #ff6b6b; color: white; 
                                  text-decoration: none; border-radius: 50px; font-size: 18px; font-weight: bold;
                                  box-shadow: 0 4px 6px rgba(0,0,0,0.1); transition: all 0.3s ease;">
                            <i class="bi bi-arrow-right-circle" style="margin-right: 8px;"></i>作品の詳細を見る
                        </a>
                    </div>
                </div>
            `;
        }
        
        // サンプル動画セクション（アフィリエイトリンク付き）
        if (selectedProduct.sample_movie && (selectedProduct.sample_movie.size_560_360 || selectedProduct.sample_movie.size_720_480)) {
            // 利用可能な最高画質の動画URLを選択
            const movieUrl = selectedProduct.sample_movie.size_720_480 || selectedProduct.sample_movie.size_644_414 || 
                            selectedProduct.sample_movie.size_560_360 || selectedProduct.sample_movie.size_476_306;
            
            productImageHtml += `
                <div class="dmm-sample-section">
                    <h3>サンプル動画</h3>
                    <div style="text-align: center;">
                        <div style="display: inline-block;">
                            <button onclick="window.open('${movieUrl}', '_blank')" 
                                    style="display: inline-block; padding: 12px 30px; background-color: #007bff; color: white; 
                                           text-decoration: none; border-radius: 5px; font-weight: bold; border: none; cursor: pointer;
                                           margin-right: 10px;">
                                <i class="bi bi-play-circle" style="margin-right: 8px;"></i>サンプル動画を再生
                            </button>
                            <a href="${selectedProduct.affiliateURL || selectedProduct.url}" target="_blank" rel="nofollow noopener" 
                               style="display: inline-block; padding: 12px 30px; background-color: #28a745; color: white; 
                                      text-decoration: none; border-radius: 5px; font-weight: bold;">
                                <i class="bi bi-cart" style="margin-right: 8px;"></i>商品ページで詳細を見る
                            </a>
                        </div>
                        <p style="margin-top: 10px; color: #666; font-size: 14px;">
                            ※ サンプル動画はポップアップウィンドウで開きます<br>
                            ※ 商品の購入・詳細確認は商品ページでお願いします
                        </p>
                    </div>
                </div>
            `;
        }
    }
    
    // 最初のh1タグの後に商品画像を挿入
    if (productImageHtml) {
        const h1Match = content.match(/(<h1[^>]*>.*?<\/h1>)/i);
        if (h1Match) {
            content = content.replace(h1Match[0], h1Match[0] + '\n' + productImageHtml);
        } else {
            // h1タグがない場合は先頭に挿入
            content = productImageHtml + '\n' + content;
        }
    }
    
    // 生成された画像を挿入（商品画像の後に）
    if (result.images && result.images.length > 0) {
        result.images.forEach((image, index) => {
            // プレースホルダー画像はスキップ
            if (image.is_placeholder || image.url.includes('dmm-placeholder')) {
                return;
            }
            
            const imageHtml = `
                <figure class="text-center my-4">
                    <img src="${image.url}" alt="${image.alt}" class="img-fluid rounded shadow">
                    <figcaption class="text-muted small mt-2">${image.alt}</figcaption>
                </figure>
            `;
            
            // h2タグの後に画像を挿入（最初のh2は商品画像の後なので、スキップしない）
            const h2Tags = content.match(/<h2[^>]*>.*?<\/h2>/gi);
            if (h2Tags && h2Tags[index]) {
                // 各h2タグの位置を保持して、順番に挿入
                let targetH2 = h2Tags[index];
                let insertPosition = content.lastIndexOf(targetH2) + targetH2.length;
                content = content.slice(0, insertPosition) + '\n' + imageHtml + content.slice(insertPosition);
            }
        });
    }
    
    // 結果を保存（グローバル変数に）
    generatedContent = styleTag + content;
    
    document.getElementById('resultContent').innerHTML = generatedContent;
    
    // メタデータの表示
    if (result.metadata) {
        const metaHtml = `
            <div class="alert alert-info mt-4">
                <small>
                    <strong>商品ID:</strong> ${result.metadata.product_id} | 
                    <strong>モデル:</strong> ${result.metadata.model} | 
                    <strong>Perplexity使用:</strong> ${result.metadata.perplexity_used ? 'あり' : 'なし'} | 
                    <strong>生成時刻:</strong> ${result.metadata.timestamp}
                </small>
            </div>
        `;
        document.getElementById('resultContent').innerHTML += metaHtml;
    }
    
    new bootstrap.Modal(document.getElementById('resultModal')).show();
}

// コピー機能
async function copyDMMArticle() {
    if (!generatedContent) return;
    
    try {
        await navigator.clipboard.writeText(generatedContent);
        showToast('クリップボードにコピーしました', 'success');
    } catch (error) {
        showToast('コピーに失敗しました', 'danger');
    }
}

// ダウンロード機能
function downloadDMMArticle() {
    if (!generatedContent || !selectedProduct) return;
    
    const html = `<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${selectedProduct.title} - レビュー記事</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 { color: #2c3e50; }
        img { max-width: 100%; height: auto; }
        figure { margin: 2em 0; text-align: center; }
        figcaption { margin-top: 0.5em; color: #666; font-size: 0.9em; }
        table { width: 100%; border-collapse: collapse; margin: 1em 0; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f5f5f5; }
        
        /* CTAボタンのスタイル */
        .dmm-cta-section {
            margin: 40px 0;
            padding: 30px;
            background-color: #f8f9fa;
            border-radius: 10px;
            text-align: center;
        }
        .dmm-cta-section h3 {
            color: #333;
            margin-bottom: 20px;
        }
        .dmm-cta-section p {
            color: #666;
            margin-bottom: 25px;
        }
        .dmm-cta-section a {
            display: inline-block;
            padding: 15px 40px;
            background-color: #ff6b6b;
            color: white;
            text-decoration: none;
            border-radius: 50px;
            font-size: 18px;
            font-weight: bold;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }
        .dmm-cta-section a:hover {
            background-color: #ff5252;
            transform: translateY(-2px);
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
    </style>
</head>
<body>
    ${generatedContent}
</body>
</html>`;
    
    const blob = new Blob([html], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${selectedProduct.title.replace(/[^\w\s]/gi, '')}_article.html`;
    a.click();
    URL.revokeObjectURL(url);
    
    showToast('ダウンロードを開始しました', 'success');
}

// プリセット保存
async function saveDMMPreset() {
    const name = prompt('プリセット名を入力してください:');
    if (!name) return;
    
    // プリセットデータを準備
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'save');
    formData.append('name', name);
    formData.append('model', 'venice-' + document.getElementById('veniceModel').value);
    formData.append('temperature', '0.7'); // デフォルト値
    formData.append('max_tokens', '3000'); // DMM用のデフォルト値
    formData.append('system_prompt', document.getElementById('systemPrompt').value);
    
    // 画像生成設定
    if (document.getElementById('enableImages').checked) {
        formData.append('enable_image', '1');
        formData.append('image_model', document.getElementById('imageModel').value);
        formData.append('image_style', document.getElementById('imageStyle').value);
        formData.append('image_prompt', document.getElementById('imagePrompt').value);
    }
    
    try {
        const response = await fetch('../api/preset-service.php', {
            method: 'POST',
            body: formData,
            credentials: 'same-origin'
        });
        
        const result = await response.json();
        
        if (result.success) {
            showToast('プリセットを保存しました', 'success');
            // ページをリロードして新しいプリセットを反映
            setTimeout(() => location.reload(), 1500);
        } else {
            showToast('保存に失敗しました: ' + result.error, 'danger');
        }
    } catch (error) {
        showToast('エラー: ' + error.message, 'danger');
    }
}

// トースト表示
function showToast(message, type = 'info') {
    const toastHtml = `
        <div class="toast align-items-center text-white bg-${type} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">${message}</div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    let container = document.querySelector('.toast-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
    }
    
    container.insertAdjacentHTML('beforeend', toastHtml);
    const toastEl = container.lastElementChild;
    const toast = new bootstrap.Toast(toastEl);
    toast.show();
    
    toastEl.addEventListener('hidden.bs.toast', () => {
        toastEl.remove();
    });
}
</script>
</div> <!-- page-content の閉じタグ -->