<?php
// 設定ファイルが読み込まれているか確認
if (!isset($config)) {
    // 直接アクセスの場合はリダイレクト
    header('Location: ../index.php?page=duga');
    exit;
}
?>

<div class="page-content" style="min-height: 100vh; padding-bottom: 100px;">
    <div class="page-header mb-4">
        <h1><i class="bi bi-film"></i> DUGA商品記事生成</h1>
        <p class="text-muted">DUGAの商品情報を取得し、AIで記事を自動生成します</p>
    </div>

    <!-- APIキー設定の確認 -->
    <div id="apiKeyAlert" class="alert alert-warning d-none">
        <i class="bi bi-exclamation-triangle"></i> 
        <span id="apiKeyMessage"></span>
        <a href="?page=settings" class="alert-link">設定画面</a>で設定してください。
    </div>

    <!-- Step 1: 商品検索 -->
    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h5 class="mb-0"><i class="bi bi-search"></i> Step 1: 商品検索</h5>
        </div>
        <div class="card-body">
            <form id="searchForm">
                <div class="row">
                    <div class="col-md-6">
                        <label for="searchKeyword" class="form-label">検索キーワード</label>
                        <input type="text" class="form-control" id="searchKeyword" placeholder="商品名、女優名、メーカー名など">
                    </div>
                    <div class="col-md-3">
                        <label for="searchSort" class="form-label">並び順</label>
                        <select class="form-select" id="searchSort">
                            <option value="new">新着順</option>
                            <option value="favorite">人気順</option>
                            <option value="release">発売日順</option>
                            <option value="price">価格順</option>
                            <option value="rating">評価順</option>
                        </select>
                    </div>
                </div>
                <div class="row mt-3">
                    <div class="col-md-12">
                        <button type="submit" class="btn btn-primary btn-lg" id="searchBtn">
                            <i class="bi bi-search"></i> 商品を検索
                        </button>
                    </div>
                </div>
            </form>
            
            <!-- 検索結果 -->
            <div id="searchResults" class="mt-4" style="display: none;">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h6 class="mb-0">検索結果 <span id="resultCount" class="text-muted"></span></h6>
                    <div id="paginationTop"></div>
                </div>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th width="100">画像</th>
                                <th>商品名</th>
                                <th>価格</th>
                                <th>評価</th>
                                <th>操作</th>
                            </tr>
                        </thead>
                        <tbody id="productList">
                            <!-- 動的に生成 -->
                        </tbody>
                    </table>
                </div>
                <div id="paginationBottom" class="mt-3 d-flex justify-content-center"></div>
            </div>
        </div>
    </div>

    <!-- Step 2: 商品詳細と記事生成設定 -->
    <div class="card mb-4" id="productDetailCard" style="display: none; margin-bottom: 100px;">
        <div class="card-header bg-success text-white">
            <h5 class="mb-0"><i class="bi bi-file-text"></i> Step 2: 記事生成設定</h5>
        </div>
        <div class="card-body" style="overflow: visible;">
            <!-- 選択された商品情報 -->
            <div id="selectedProduct" class="mb-4 p-3 bg-light rounded">
                <!-- 動的に生成 -->
            </div>
            
            <!-- 記事生成設定 -->
            <form id="generateForm">
                <!-- プリセット選択 -->
                <div class="mb-3">
                    <label for="presetSelect" class="form-label">プリセット選択</label>
                    <div class="input-group">
                        <select class="form-select" id="presetSelect" onchange="loadPreset()">
                            <option value="">-- プリセットを選択 --</option>
                            <?php
                            // プリセットファイルの読み込み
                            $presetsFile = __DIR__ . '/../../data/presets.json';
                            $presets = [];
                            if (file_exists($presetsFile)) {
                                $presets = json_decode(file_get_contents($presetsFile), true) ?: [];
                            }
                            foreach ($presets as $key => $preset): 
                            ?>
                                <option value="<?php echo htmlspecialchars($key); ?>" 
                                        data-model="<?php echo htmlspecialchars($preset['model'] ?? ''); ?>"
                                        data-temperature="<?php echo htmlspecialchars($preset['temperature'] ?? 0.7); ?>"
                                        data-max-tokens="<?php echo htmlspecialchars($preset['max_tokens'] ?? 2000); ?>">
                                    <?php echo htmlspecialchars($preset['name']); ?>
                                </option>
                            <?php 
                            endforeach; 
                            ?>
                        </select>
                        <button type="button" class="btn btn-outline-secondary" onclick="clearPreset()">
                            <i class="bi bi-x-circle"></i> クリア
                        </button>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="aiModel" class="form-label">AIモデル</label>
                        <select class="form-select" id="aiModel">
                            <?php foreach ($config['models'] as $provider => $models): ?>
                                <optgroup label="<?php echo ucfirst($provider); ?>">
                                    <?php foreach ($models as $key => $model): ?>
                                        <option value="<?php echo htmlspecialchars($key); ?>" 
                                                <?php echo $key === $config['default_model'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($model['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </optgroup>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">情報収集オプション</label>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="enablePerplexity" checked>
                            <label class="form-check-label" for="enablePerplexity">
                                Perplexityで追加情報を収集する
                            </label>
                        </div>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label for="systemPrompt" class="form-label">システムプロンプト</label>
                    <textarea class="form-control" id="systemPrompt" rows="10" placeholder="AIに与える指示を記述します。プリセットを選択すると自動的に設定されます。"></textarea>
                    <small class="text-muted">プリセットを選択すると、プリセットに設定されたシステムプロンプトが自動的に読み込まれます</small>
                </div>
                
                <div class="mb-3">
                    <label for="keywords" class="form-label">キーワード</label>
                    <input type="text" class="form-control" id="keywords" placeholder="SEO用のキーワード（カンマ区切り）">
                    <small class="text-muted">記事に含めたいキーワードをカンマ区切りで入力</small>
                </div>
                
                <div class="mb-3">
                    <label for="additionalInfo" class="form-label">追加情報（任意）</label>
                    <textarea class="form-control" id="additionalInfo" rows="3" placeholder="記事に含めたい追加情報があれば入力してください"></textarea>
                </div>
                
                <!-- 画像・動画自動挿入設定 -->
                <div class="card mb-3">
                    <div class="card-header">
                        <h6><i class="bi bi-images"></i> 画像・動画の自動挿入設定</h6>
                    </div>
                    <div class="card-body">
                        <div class="form-check mb-2">
                            <input type="checkbox" class="form-check-input" id="autoInsertImages">
                            <label class="form-check-label" for="autoInsertImages">
                                サンプル画像を最初の見出し（h2タグ）後に自動挿入
                            </label>
                            <small class="text-muted d-block ms-4">記事生成時に商品のサンプル画像を自動的に挿入します</small>
                        </div>
                    </div>
                </div>
                
                <!-- 画像生成設定 -->
                <div class="card mb-3">
                    <div class="card-header">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="enableImages" checked>
                            <label class="form-check-label" for="enableImages">
                                <strong>画像生成を有効にする</strong>
                            </label>
                        </div>
                    </div>
                    <div class="card-body" id="imageSettings">
                        <div class="row">
                            <div class="col-md-6">
                                <label for="imageModel" class="form-label">画像生成モデル</label>
                                <select class="form-select" id="imageModel">
                                    <?php foreach ($config['image_models'] as $key => $model): ?>
                                        <option value="<?php echo htmlspecialchars($key); ?>">
                                            <?php echo htmlspecialchars($model['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label for="imageStyle" class="form-label">画像スタイル</label>
                                <select class="form-select" id="imageStyle">
                                    <?php foreach ($config['image_styles'] as $key => $style): ?>
                                        <option value="<?php echo htmlspecialchars($key); ?>">
                                            <?php echo htmlspecialchars($style); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label for="numImages" class="form-label">画像数</label>
                                <input type="number" class="form-control" id="numImages" min="1" max="10" value="3">
                            </div>
                        </div>
                        <div class="mt-3">
                            <label for="imagePrompt" class="form-label">画像生成プロンプト</label>
                            <textarea class="form-control" id="imagePrompt" rows="2" placeholder="画像生成の基本プロンプト（オプション）"></textarea>
                            <small class="text-muted">空欄の場合は記事内容に基づいて自動生成されます</small>
                        </div>
                    </div>
                </div>
                
                <div class="form-check mb-3">
                    <input class="form-check-input" type="checkbox" id="saveHistory" checked>
                    <label class="form-check-label" for="saveHistory">
                        履歴に保存
                    </label>
                </div>
                
                <button type="submit" class="btn btn-success btn-lg" id="generateBtn">
                    <i class="bi bi-magic"></i> 記事を生成
                </button>
            </form>
        </div>
    </div>
</div>

<!-- 結果表示モーダル -->
<div class="modal fade" id="resultModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="true">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">生成結果</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" style="max-height: calc(100vh - 200px); overflow-y: auto;">
                <div id="resultContent"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="copyDUGAArticle()">
                    <i class="bi bi-clipboard"></i> コピー
                </button>
                <button type="button" class="btn btn-primary" onclick="downloadDUGAArticle()">
                    <i class="bi bi-download"></i> ダウンロード
                </button>
                <button type="button" class="btn btn-success" onclick="saveDUGAPreset()">
                    <i class="bi bi-bookmark"></i> プリセット保存
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// グローバル変数
let selectedProduct = null;
let generatedContent = null;
let currentSearchParams = null;
let currentPage = 1;
let totalCount = 0;
const itemsPerPage = 20;
let presets = <?php echo json_encode($presets); ?>;

// ページ読み込み時の処理
document.addEventListener('DOMContentLoaded', function() {
    checkAPIKeys();
    
    // 画像設定の表示切替
    document.getElementById('enableImages').addEventListener('change', function() {
        document.getElementById('imageSettings').style.display = this.checked ? 'block' : 'none';
    });
});

// APIキーの確認
function checkAPIKeys() {
    const requiredKeys = [];
    const config = <?php echo json_encode([
        'duga_app_id' => !empty($config['duga']['app_id']),
        'duga_agent_id' => !empty($config['duga']['agent_id']),
        'duga_banner_id' => !empty($config['duga']['banner_id']),
        'has_any_ai_key' => !empty($config['api_keys']['openai']) || 
                           !empty($config['api_keys']['anthropic']) || 
                           !empty($config['api_keys']['claude']) || 
                           !empty($config['api_keys']['google']) || 
                           !empty($config['api_keys']['gemini']) || 
                           !empty($config['api_keys']['venice']) ||
                           !empty($config['api_keys']['perplexity']) ||
                           !empty($config['openai_api_key']) ||
                           !empty($config['anthropic_api_key']) ||
                           !empty($config['google_api_key']) ||
                           !empty($config['venice_api_key']) ||
                           !empty($config['perplexity_api_key'])
    ]); ?>;
    
    if (!config.duga_app_id || !config.duga_agent_id || !config.duga_banner_id) {
        requiredKeys.push('DUGA API');
    }
    
    // いずれかのAIキーが設定されているかチェック
    if (!config.has_any_ai_key) {
        requiredKeys.push('AI API（OpenAI、Anthropic、Google、Venice、Perplexityのいずれか）');
    }
    
    if (requiredKeys.length > 0) {
        document.getElementById('apiKeyAlert').classList.remove('d-none');
        document.getElementById('apiKeyMessage').textContent = 
            requiredKeys.join('、') + 'のキーが設定されていません。';
    } else {
        // すべてのキーが設定されている場合は警告を非表示
        document.getElementById('apiKeyAlert').classList.add('d-none');
    }
}

// 商品検索フォーム
document.getElementById('searchForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    
    // 検索パラメータを保存（空白検索も許可）
    currentSearchParams = {
        keyword: document.getElementById('searchKeyword').value.trim(), // 空白も許可
        sort: document.getElementById('searchSort').value
    };
    
    // 新しい検索の場合は1ページ目から
    currentPage = 1;
    
    // 検索実行
    await searchProducts(currentPage);
});

// 商品検索実行
async function searchProducts(page = 1) {
    const btn = document.getElementById('searchBtn');
    const originalText = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>検索中...';
    
    try {
        const offset = (page - 1) * itemsPerPage + 1;
        
        const response = await fetch('../api/duga-service.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'search_products',
                ...currentSearchParams,
                hits: itemsPerPage,
                offset: offset
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            // デバッグ情報がある場合はコンソールに表示
            if (result.debug) {
                console.log('DUGA API Debug:', result.debug);
            }
            
            totalCount = result.result.total_count || 0;
            currentPage = page;
            console.log('DUGA Search Results:', result.result.items); // デバッグ用
            displaySearchResults(result.result.items);
            updatePagination();
            document.getElementById('searchResults').style.display = 'block';
            
            // 結果件数を表示
            document.getElementById('resultCount').textContent = 
                `（全${totalCount}件中 ${offset}〜${Math.min(offset + itemsPerPage - 1, totalCount)}件を表示）`;
        } else {
            showToast('検索エラー: ' + result.error, 'danger');
        }
        
    } catch (error) {
        showToast('通信エラー: ' + error.message, 'danger');
    } finally {
        btn.disabled = false;
        btn.innerHTML = originalText;
    }
}

// 検索結果の表示
function displaySearchResults(products) {
    const tbody = document.getElementById('productList');
    tbody.innerHTML = '';
    
    if (products.length === 0) {
        tbody.innerHTML = '<tr><td colspan="5" class="text-center">商品が見つかりませんでした</td></tr>';
        return;
    }
    
    products.forEach(product => {
        const tr = document.createElement('tr');
        tr.innerHTML = `
            <td>
                <img src="${product.imageURL || 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="80" height="80" viewBox="0 0 80 80"%3E%3Crect width="80" height="80" fill="%23e0e0e0"/%3E%3Ctext x="50%25" y="50%25" text-anchor="middle" dy=".3em" fill="%23999" font-size="12"%3ENo Image%3C/text%3E%3C/svg%3E'}" 
                     alt="${product.title}" 
                     class="img-thumbnail" 
                     style="max-width: 80px;"
                     onerror="this.src='data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 width=%2280%22 height=%2280%22 viewBox=%220 0 80 80%22%3E%3Crect width=%2280%22 height=%2280%22 fill=%22%23e0e0e0%22/%3E%3Ctext x=%2250%25%22 y=%2250%25%22 text-anchor=%22middle%22 dy=%22.3em%22 fill=%22%23999%22 font-size=%2212%22%3ENo Image%3C/text%3E%3C/svg%3E'">
            </td>
            <td>
                <strong>${product.title}</strong>
                <br><small class="text-muted">${product.maker_name || '-'}</small>
            </td>
            <td>¥${product.list_price || product.price || '-'}</td>
            <td>
                ${product.review_average > 0 ? 
                    `<span class="text-warning">★${product.review_average}</span> (${product.review_count})` : 
                    '-'
                }
            </td>
            <td>
                <button class="btn btn-sm btn-primary" onclick='selectProduct(${JSON.stringify(product).replace(/'/g, "\\'")})'>
                    選択
                </button>
            </td>
        `;
        tbody.appendChild(tr);
    });
}

// ページネーションの更新
function updatePagination() {
    const totalPages = Math.ceil(totalCount / itemsPerPage);
    
    // ページネーションが必要ない場合は非表示
    if (totalPages <= 1) {
        document.getElementById('paginationTop').innerHTML = '';
        document.getElementById('paginationBottom').innerHTML = '';
        return;
    }
    
    // ページネーションHTML生成
    let paginationHtml = '<nav><ul class="pagination mb-0">';
    
    // 前へボタン
    if (currentPage > 1) {
        paginationHtml += `<li class="page-item">
            <a class="page-link" href="#" onclick="changePage(${currentPage - 1}); return false;">
                <i class="bi bi-chevron-left"></i> 前へ
            </a>
        </li>`;
    } else {
        paginationHtml += `<li class="page-item disabled">
            <span class="page-link"><i class="bi bi-chevron-left"></i> 前へ</span>
        </li>`;
    }
    
    // ページ番号
    const startPage = Math.max(1, currentPage - 2);
    const endPage = Math.min(totalPages, startPage + 4);
    
    if (startPage > 1) {
        paginationHtml += `<li class="page-item">
            <a class="page-link" href="#" onclick="changePage(1); return false;">1</a>
        </li>`;
        if (startPage > 2) {
            paginationHtml += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
        }
    }
    
    for (let i = startPage; i <= endPage; i++) {
        if (i === currentPage) {
            paginationHtml += `<li class="page-item active">
                <span class="page-link">${i}</span>
            </li>`;
        } else {
            paginationHtml += `<li class="page-item">
                <a class="page-link" href="#" onclick="changePage(${i}); return false;">${i}</a>
            </li>`;
        }
    }
    
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            paginationHtml += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
        }
        paginationHtml += `<li class="page-item">
            <a class="page-link" href="#" onclick="changePage(${totalPages}); return false;">${totalPages}</a>
        </li>`;
    }
    
    // 次へボタン
    if (currentPage < totalPages) {
        paginationHtml += `<li class="page-item">
            <a class="page-link" href="#" onclick="changePage(${currentPage + 1}); return false;">
                次へ <i class="bi bi-chevron-right"></i>
            </a>
        </li>`;
    } else {
        paginationHtml += `<li class="page-item disabled">
            <span class="page-link">次へ <i class="bi bi-chevron-right"></i></span>
        </li>`;
    }
    
    paginationHtml += '</ul></nav>';
    
    // 上部と下部に設置
    document.getElementById('paginationTop').innerHTML = paginationHtml;
    document.getElementById('paginationBottom').innerHTML = paginationHtml;
}

// ページ変更
async function changePage(page) {
    if (page < 1 || page === currentPage) return;
    
    // スクロール位置を検索結果の上に
    document.getElementById('searchResults').scrollIntoView({ behavior: 'smooth', block: 'start' });
    
    // 検索実行
    await searchProducts(page);
}

// 商品選択
function selectProduct(product) {
    selectedProduct = product;
    
    // サンプル画像のHTML生成
    let sampleImagesHtml = '';
    if (product.sampleImageURL && product.sampleImageURL.length > 0) {
        sampleImagesHtml = '<div class="mt-3"><h6>サンプル画像:</h6><div class="row">';
        product.sampleImageURL.slice(0, 6).forEach((img, index) => {
            sampleImagesHtml += `
                <div class="col-md-4 mb-2">
                    <img src="${img}" alt="サンプル画像${index + 1}" 
                         class="img-fluid rounded" 
                         style="cursor: pointer;" 
                         onclick="window.open('${img}', '_blank')">
                </div>
            `;
        });
        sampleImagesHtml += '</div></div>';
    }
    
    // 商品詳細を表示
    document.getElementById('selectedProduct').innerHTML = `
        <div class="row">
            <div class="col-md-2">
                <img src="${product.imageURL || 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="150" height="150" viewBox="0 0 150 150"%3E%3Crect width="150" height="150" fill="%23e0e0e0"/%3E%3Ctext x="50%25" y="50%25" text-anchor="middle" dy=".3em" fill="%23999"%3ENo Image%3C/text%3E%3C/svg%3E'}" 
                     alt="${product.title}" 
                     class="img-fluid rounded"
                     onerror="this.src='data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 width=%22150%22 height=%22150%22 viewBox=%220 0 150 150%22%3E%3Crect width=%22150%22 height=%22150%22 fill=%22%23e0e0e0%22/%3E%3Ctext x=%2250%25%22 y=%2250%25%22 text-anchor=%22middle%22 dy=%22.3em%22 fill=%22%23999%22%3ENo Image%3C/text%3E%3C/svg%3E'">
            </div>
            <div class="col-md-10">
                <h5>${product.title}</h5>
                <p class="mb-1"><strong>価格:</strong> ¥${product.list_price || product.price || '-'}</p>
                <p class="mb-1"><strong>メーカー:</strong> ${product.maker_name || '-'}</p>
                ${product.content_id ? `<p class="mb-1"><strong>商品ID:</strong> ${product.content_id}</p>` : ''}
                ${product.iteminfo && product.iteminfo.actress && product.iteminfo.actress.length > 0 ? 
                    `<p class="mb-1"><strong>出演:</strong> ${product.iteminfo.actress.map(a => a.name).join(', ')}</p>` : ''}
                ${product.iteminfo && product.iteminfo.genre && product.iteminfo.genre.length > 0 ? 
                    `<p class="mb-1"><strong>ジャンル:</strong> ${product.iteminfo.genre.map(g => g.name).join(', ')}</p>` : ''}
                ${product.review_average > 0 ? `<p class="mb-1"><strong>評価:</strong> <span class="text-warning">★${product.review_average}</span> (${product.review_count}件)</p>` : ''}
            </div>
        </div>
        ${sampleImagesHtml}
    `;
    
    document.getElementById('productDetailCard').style.display = 'block';
    
    // スクロールを少し遅延させて確実に表示
    setTimeout(() => {
        const detailCard = document.getElementById('productDetailCard');
        detailCard.scrollIntoView({ behavior: 'smooth', block: 'start' });
        
        // さらに少し上にスクロールして余白を確保
        setTimeout(() => {
            window.scrollBy(0, -100);
        }, 500);
    }, 100);
}

// プリセット読み込み
function loadPreset() {
    const presetKey = document.getElementById('presetSelect').value;
    if (!presetKey) return;
    
    const preset = presets[presetKey];
    if (!preset) return;
    
    // モデルの設定
    if (preset.model) {
        const modelId = preset.model.startsWith('venice-') ? preset.model.replace('venice-', '') : preset.model;
        const modelSelect = document.getElementById('aiModel');
        if (modelSelect) {
            const optionExists = Array.from(modelSelect.options).some(option => option.value === modelId);
            if (optionExists) {
                modelSelect.value = modelId;
            } else {
                modelSelect.selectedIndex = 0;
            }
        }
    }
    
    // システムプロンプトの設定
    if (preset.system_prompt) {
        document.getElementById('systemPrompt').value = preset.system_prompt;
    }
    
    // 画像生成設定
    if (typeof preset.enable_image !== 'undefined') {
        document.getElementById('enableImages').checked = preset.enable_image;
        document.getElementById('imageSettings').style.display = preset.enable_image ? 'block' : 'none';
    }
    
    if (preset.image_model) {
        document.getElementById('imageModel').value = preset.image_model;
    }
    
    if (preset.image_style) {
        document.getElementById('imageStyle').value = preset.image_style;
    }
    
    if (preset.image_prompt) {
        document.getElementById('imagePrompt').value = preset.image_prompt;
    }
}

// プリセットクリア
function clearPreset() {
    document.getElementById('presetSelect').value = '';
    document.getElementById('systemPrompt').value = '';
    document.getElementById('aiModel').selectedIndex = 0;
    document.getElementById('enableImages').checked = true;
    document.getElementById('imageSettings').style.display = 'block';
}

// 記事生成フォーム
let isGenerating = false; // 重複送信防止フラグ
document.getElementById('generateForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    
    // 既に生成中の場合は処理しない
    if (isGenerating) {
        console.log('既に生成処理中です');
        return;
    }
    
    if (!selectedProduct) {
        showToast('商品を選択してください', 'warning');
        return;
    }
    
    const btn = document.getElementById('generateBtn');
    const originalText = btn.innerHTML;
    
    // 重複送信防止
    isGenerating = true;
    btn.disabled = true;
    btn.style.pointerEvents = 'none'; // クリックイベント自体を無効化
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>生成中...';
    
    try {
        const response = await fetch('../api/duga-service.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'generate_duga_article',
                product_detail: selectedProduct,
                presets: presets,
                selected_preset: document.getElementById('presetSelect').value || 'default',
                keywords: document.getElementById('keywords').value,
                system_prompt: document.getElementById('systemPrompt').value,
                additional_info: document.getElementById('additionalInfo').value,
                ai_model: document.getElementById('aiModel').value,
                enable_perplexity: document.getElementById('enablePerplexity').checked,
                save_history: document.getElementById('saveHistory').checked,
                enable_images: document.getElementById('enableImages').checked,
                image_model: document.getElementById('imageModel').value,
                image_style: document.getElementById('imageStyle').value,
                num_images: parseInt(document.getElementById('numImages').value),
                image_prompt: document.getElementById('imagePrompt').value,
                auto_insert_images: document.getElementById('autoInsertImages').checked  // 自動挿入フラグを追加
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            generatedContent = result.content;
            showResult(result);
        } else {
            showToast('生成エラー: ' + result.error, 'danger');
        }
        
    } catch (error) {
        showToast('通信エラー: ' + error.message, 'danger');
    } finally {
        // 少し遅延してからボタンを有効化（連続クリック防止）
        setTimeout(() => {
            btn.disabled = false;
            btn.style.pointerEvents = 'auto';
            btn.innerHTML = originalText;
            isGenerating = false;
        }, 1000);
    }
});

// 結果表示
function showResult(result) {
    // 結果をグローバル変数に保存
    generatedContent = result.content || result.article;
    
    // モーダルに表示
    document.getElementById('resultContent').innerHTML = generatedContent;
    
    // メタデータの表示
    if (result.metadata) {
        const metaHtml = `
            <div class="alert alert-info mt-4">
                <small>
                    <strong>商品ID:</strong> ${result.metadata.product_id} | 
                    <strong>モデル:</strong> ${result.metadata.model} | 
                    <strong>Perplexity使用:</strong> ${result.metadata.perplexity_used ? 'あり' : 'なし'} | 
                    <strong>生成時刻:</strong> ${result.metadata.timestamp}
                </small>
            </div>
        `;
        document.getElementById('resultContent').innerHTML += metaHtml;
    }
    
    // モーダルを表示
    const resultModal = new bootstrap.Modal(document.getElementById('resultModal'));
    resultModal.show();
}

// エディタに挿入する関数（子フレームから呼ばれる）
window.insertToEditor = function(html) {
    // 現在のコンテンツに追加
    if (generatedContent) {
        generatedContent += '\n' + html;
        document.getElementById('resultContent').innerHTML = generatedContent;
        showToast('画像を挿入しました', 'success');
    }
};

// コピー機能
async function copyDUGAArticle() {
    if (!generatedContent) return;
    
    try {
        await navigator.clipboard.writeText(generatedContent);
        showToast('クリップボードにコピーしました', 'success');
    } catch (error) {
        showToast('コピーに失敗しました', 'danger');
    }
}

// ダウンロード機能
function downloadDUGAArticle() {
    if (!generatedContent || !selectedProduct) return;
    
    const html = `<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${selectedProduct.title} - レビュー記事</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 { color: #2c3e50; }
        img { max-width: 100%; height: auto; }
        figure { margin: 2em 0; text-align: center; }
        figcaption { margin-top: 0.5em; color: #666; font-size: 0.9em; }
        table { width: 100%; border-collapse: collapse; margin: 1em 0; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f5f5f5; }
        
        /* CTAボタンのスタイル */
        .duga-cta-section {
            margin: 40px 0;
            padding: 30px;
            background-color: #f8f9fa;
            border-radius: 10px;
            text-align: center;
        }
        .duga-cta-section h3 {
            color: #333;
            margin-bottom: 20px;
        }
        .duga-cta-section p {
            color: #666;
            margin-bottom: 25px;
        }
        .duga-cta-section a {
            display: inline-block;
            padding: 15px 40px;
            background-color: #ff6b6b;
            color: white;
            text-decoration: none;
            border-radius: 50px;
            font-size: 18px;
            font-weight: bold;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }
        .duga-cta-section a:hover {
            background-color: #ff5252;
            transform: translateY(-2px);
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
    </style>
</head>
<body>
    ${generatedContent}
</body>
</html>`;
    
    const blob = new Blob([html], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${selectedProduct.title.replace(/[^\w\s]/gi, '')}_article.html`;
    a.click();
    URL.revokeObjectURL(url);
    
    showToast('ダウンロードを開始しました', 'success');
}

// プリセット保存
async function saveDUGAPreset() {
    const name = prompt('プリセット名を入力してください:');
    if (!name) return;
    
    // プリセットデータを準備
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'save');
    formData.append('name', name);
    formData.append('model', document.getElementById('aiModel').value);
    formData.append('temperature', '0.7');
    formData.append('max_tokens', '3000');
    formData.append('system_prompt', document.getElementById('systemPrompt').value);
    
    // 画像生成設定
    if (document.getElementById('enableImages').checked) {
        formData.append('enable_image', '1');
        formData.append('image_model', document.getElementById('imageModel').value);
        formData.append('image_style', document.getElementById('imageStyle').value);
        formData.append('image_prompt', document.getElementById('imagePrompt').value);
    }
    
    try {
        const response = await fetch('../api/preset-service.php', {
            method: 'POST',
            body: formData,
            credentials: 'same-origin'
        });
        
        const result = await response.json();
        
        if (result.success) {
            showToast('プリセットを保存しました', 'success');
            setTimeout(() => location.reload(), 1500);
        } else {
            showToast('保存に失敗しました: ' + result.error, 'danger');
        }
    } catch (error) {
        showToast('エラー: ' + error.message, 'danger');
    }
}

// トースト表示
function showToast(message, type = 'info') {
    const toastHtml = `
        <div class="toast align-items-center text-white bg-${type} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">${message}</div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    let container = document.querySelector('.toast-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
    }
    
    container.insertAdjacentHTML('beforeend', toastHtml);
    const toastEl = container.lastElementChild;
    const toast = new bootstrap.Toast(toastEl);
    toast.show();
    
    toastEl.addEventListener('hidden.bs.toast', () => {
        toastEl.remove();
    });
}
</script>