<?php
// セッションチェックは親ファイル（index.php）で行われているため、ここでは不要

// config.phpからFTP設定を読み込み
$configFile = __DIR__ . '/../../config.php';
$config = [];
if (file_exists($configFile)) {
    $config = require $configFile;
}

// 複数のFTP設定を管理
$ftpProfiles = $config['ftp_profiles'] ?? [];

// 削除処理
if (isset($_GET['delete']) && isset($ftpProfiles[$_GET['delete']])) {
    $deletedName = $ftpProfiles[$_GET['delete']]['name'];
    unset($ftpProfiles[$_GET['delete']]);
    $config['ftp_profiles'] = $ftpProfiles;
    $configContent = "<?php\nreturn " . var_export($config, true) . ";\n";
    file_put_contents($configFile, $configContent);
    $_SESSION['ftp_message'] = "「{$deletedName}」を削除しました";
    $_SESSION['ftp_message_type'] = 'success';
    header('Location: ?page=ftp-settings');
    exit;
}
?>

<div class="container mt-4">
    <h2>FTP設定</h2>
    
    <?php if (isset($_SESSION['ftp_message'])): ?>
    <div class="alert alert-<?php echo $_SESSION['ftp_message_type'] ?? 'info'; ?> alert-dismissible fade show" role="alert">
        <?php echo htmlspecialchars($_SESSION['ftp_message']); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php
    unset($_SESSION['ftp_message']);
    unset($_SESSION['ftp_message_type']);
    ?>
    <?php endif; ?>
    
    <div class="alert alert-info">
        <i class="bi bi-info-circle"></i> 複数のFTP設定を保存して、デプロイ時に選択できます。
    </div>
    
    <!-- 保存済みのFTP設定一覧 -->
    <?php if (!empty($ftpProfiles)): ?>
    <div class="card mb-4">
        <div class="card-body">
            <h4>保存済みのFTP設定</h4>
            <div class="table-responsive">
                <table class="table table-sm">
                    <thead>
                        <tr>
                            <th>名称</th>
                            <th>ホスト</th>
                            <th>ユーザー名</th>
                            <th>リモートパス</th>
                            <th style="width: 150px;">操作</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($ftpProfiles as $id => $profile): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($profile['name']); ?></td>
                            <td><?php echo htmlspecialchars($profile['host']); ?></td>
                            <td><?php echo htmlspecialchars($profile['username']); ?></td>
                            <td><code><?php echo htmlspecialchars($profile['remote_path']); ?></code></td>
                            <td>
                                <button class="btn btn-sm btn-outline-primary" onclick="editProfile('<?php echo $id; ?>')">
                                    <i class="bi bi-pencil"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-info" onclick="testProfile('<?php echo $id; ?>')">
                                    <i class="bi bi-wifi"></i>
                                </button>
                                <a href="?page=ftp-settings&delete=<?php echo $id; ?>" 
                                   class="btn btn-sm btn-outline-danger" 
                                   onclick="return confirm('この設定を削除しますか？')">
                                    <i class="bi bi-trash"></i>
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- 新規/編集フォーム -->
    <div class="card">
        <div class="card-body">
            <h4 id="formTitle">新規FTP設定</h4>
            <form id="ftpSettingsForm">
                <input type="hidden" name="profile_id" id="profileId" value="">
                
                <div class="mb-3">
                    <label class="form-label">設定名称 <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" name="name" id="profileName" 
                           placeholder="本番サーバー" required>
                    <small class="form-text text-muted">この設定を識別するための名前</small>
                </div>
                
                <div class="row mb-3">
                    <div class="col-12 col-md-8 mb-3 mb-md-0">
                        <label class="form-label">FTPホスト <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" name="host" id="ftpHost"
                               placeholder="ftp.example.com" required>
                        <small class="form-text text-muted">FTPサーバーのホスト名またはIPアドレス</small>
                    </div>
                    <div class="col-12 col-md-4">
                        <label class="form-label">ポート</label>
                        <input type="number" class="form-control" name="port" id="ftpPort"
                               value="21" min="1" max="65535" required>
                        <small class="form-text text-muted">通常は21</small>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-12 col-md-6 mb-3 mb-md-0">
                        <label class="form-label">ユーザー名 <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" name="username" id="ftpUsername"
                               placeholder="ftpuser" required>
                    </div>
                    <div class="col-12 col-md-6">
                        <label class="form-label">パスワード <span class="text-danger">*</span></label>
                        <input type="password" class="form-control" name="password" id="ftpPassword"
                               placeholder="••••••••" required>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">リモートパス</label>
                    <input type="text" class="form-control" name="remote_path" id="ftpRemotePath"
                           placeholder="/public_html/" value="/">
                    <small class="form-text text-muted">
                        アップロード先のディレクトリパス。ドメイン直下の場合は「/」、
                        サブディレクトリの場合は「/public_html/blog/」など
                    </small>
                </div>
                
                <div class="row mb-3">
                    <div class="col-12 col-md-6 mb-3 mb-md-0">
                        <div class="form-check">
                            <input type="checkbox" class="form-check-input" name="passive" 
                                   id="passiveMode" checked>
                            <label class="form-check-label" for="passiveMode">
                                パッシブモードを使用
                            </label>
                            <small class="form-text text-muted d-block">通常はチェックを推奨</small>
                        </div>
                    </div>
                    <div class="col-12 col-md-6">
                        <div class="form-check">
                            <input type="checkbox" class="form-check-input" name="ssl" 
                                   id="sslMode">
                            <label class="form-check-label" for="sslMode">
                                SSL/TLS接続を使用（FTPS）
                            </label>
                            <small class="form-text text-muted d-block">対応している場合はチェック</small>
                        </div>
                    </div>
                </div>
                
                <div class="alert alert-warning">
                    <i class="bi bi-shield-lock"></i> <strong>セキュリティ注意:</strong> 
                    パスワードは暗号化して保存されますが、完全なセキュリティは保証されません。
                    信頼できる環境でのみご使用ください。
                </div>
                
                <div class="d-inline-flex gap-2">
                    <button type="button" class="btn btn-secondary" id="testConnection">
                        <i class="bi bi-wifi"></i> 接続テスト
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-save"></i> 保存
                    </button>
                    <button type="button" class="btn btn-outline-secondary" id="cancelEdit" style="display: none;">
                        <i class="bi bi-x"></i> キャンセル
                    </button>
                </div>
                
                <div id="ftpStatus" class="mt-3"></div>
            </form>
        </div>
    </div>
    
    <!-- FTPトラブルシューティング -->
    <div class="card mt-4">
        <div class="card-body">
            <h4>FTP接続トラブルシューティング</h4>
            <div class="accordion" id="ftpTroubleshooting">
                <div class="accordion-item">
                    <h2 class="accordion-header">
                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#troubleshoot1">
                            「FTPサーバーに接続できません」エラー
                        </button>
                    </h2>
                    <div id="troubleshoot1" class="accordion-collapse collapse" data-bs-parent="#ftpTroubleshooting">
                        <div class="accordion-body">
                            <ul>
                                <li><strong>ホスト名を確認</strong>: ftp://を含めずに入力（例: ftp.example.com）</li>
                                <li><strong>ポート番号を確認</strong>: 通常は21、FTPS（SSL）の場合は990の場合も</li>
                                <li><strong>ファイアウォール</strong>: ポート21（または指定ポート）が開いているか確認</li>
                                <li><strong>IPアドレスで試す</strong>: ホスト名の代わりにIPアドレスを使用</li>
                            </ul>
                        </div>
                    </div>
                </div>
                <div class="accordion-item">
                    <h2 class="accordion-header">
                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#troubleshoot2">
                            「ログインに失敗しました」エラー
                        </button>
                    </h2>
                    <div id="troubleshoot2" class="accordion-collapse collapse" data-bs-parent="#ftpTroubleshooting">
                        <div class="accordion-body">
                            <ul>
                                <li><strong>ユーザー名/パスワード</strong>: 大文字小文字を確認、前後の空白を削除</li>
                                <li><strong>FTPアカウント</strong>: Webログインとは別のFTP専用アカウントの場合があります</li>
                                <li><strong>アカウントロック</strong>: 複数回失敗するとロックされる場合があります</li>
                                <li><strong>特殊文字</strong>: パスワードに特殊文字が含まれる場合、エスケープが必要な場合があります</li>
                            </ul>
                        </div>
                    </div>
                </div>
                <div class="accordion-item">
                    <h2 class="accordion-header">
                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#troubleshoot3">
                            「FTP操作に失敗しました」エラー
                        </button>
                    </h2>
                    <div id="troubleshoot3" class="accordion-collapse collapse" data-bs-parent="#ftpTroubleshooting">
                        <div class="accordion-body">
                            <ul>
                                <li><strong>パッシブモード</strong>: チェックを入れる/外すを試してください</li>
                                <li><strong>SSL/TLS設定</strong>: サーバーがFTPSに対応していない場合はチェックを外す</li>
                                <li><strong>ディレクトリ権限</strong>: FTPユーザーに適切な権限があるか確認</li>
                                <li><strong>リモートパス</strong>: 正しいパスか確認（/public_html/、/www/など）</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// 保存済みプロファイルデータ
const ftpProfiles = <?php echo json_encode($ftpProfiles); ?>;

$(document).ready(function() {
    // 新規追加されたアイテムをハイライト
    const newId = sessionStorage.getItem('ftp_new_id');
    const newName = sessionStorage.getItem('ftp_new_name');
    if (newId && newName) {
        // 該当する行を探してハイライト
        $('tbody tr').each(function() {
            if ($(this).find('td').first().text().trim() === newName) {
                $(this).addClass('highlight-new');
                // スクロール
                $('html, body').animate({
                    scrollTop: $(this).offset().top - 100
                }, 500);
            }
        });
        
        // セッションストレージをクリア
        sessionStorage.removeItem('ftp_new_id');
        sessionStorage.removeItem('ftp_new_name');
    }
    // プロファイル編集
    window.editProfile = function(id) {
        const profile = ftpProfiles[id];
        if (!profile) return;
        
        $('#formTitle').text('FTP設定を編集');
        $('#profileId').val(id);
        $('#profileName').val(profile.name);
        $('#ftpHost').val(profile.host);
        $('#ftpPort').val(profile.port);
        $('#ftpUsername').val(profile.username);
        $('#ftpPassword').val(atob(profile.password));
        $('#ftpRemotePath').val(profile.remote_path);
        $('#passiveMode').prop('checked', profile.passive);
        $('#sslMode').prop('checked', profile.ssl);
        $('#cancelEdit').show();
        
        // フォームまでスクロール
        $('html, body').animate({
            scrollTop: $('#ftpSettingsForm').offset().top - 100
        }, 500);
    };
    
    // プロファイルテスト
    window.testProfile = function(id) {
        const profile = ftpProfiles[id];
        if (!profile) return;
        
        const btn = $(`button[onclick="testProfile('${id}')"]`);
        const originalHtml = btn.html();
        btn.prop('disabled', true).html('<i class="bi bi-arrow-repeat spin"></i>');
        
        $.ajax({
            url: '/seitekisitebilder/api/test-ftp.php',
            method: 'POST',
            data: {
                host: profile.host,
                port: profile.port,
                username: profile.username,
                password: atob(profile.password),
                passive: profile.passive ? 'on' : '',
                ssl: profile.ssl ? 'on' : '',
                remote_path: profile.remote_path
            },
            success: function(response) {
                if (response.success) {
                    alert('接続成功！');
                } else {
                    alert('接続失敗: ' + response.error);
                }
            },
            error: function() {
                alert('テスト中にエラーが発生しました。');
            },
            complete: function() {
                btn.prop('disabled', false).html(originalHtml);
            }
        });
    };
    
    // キャンセルボタン
    $('#cancelEdit').click(function() {
        $('#formTitle').text('新規FTP設定');
        $('#ftpSettingsForm')[0].reset();
        $('#profileId').val('');
        $(this).hide();
    });
    
    // 接続テスト
    $('#testConnection').click(function() {
        const btn = $(this);
        const originalText = btn.html();
        btn.prop('disabled', true).html('<i class="bi bi-arrow-repeat spin"></i> テスト中...');
        
        $('#ftpStatus').html('<div class="alert alert-info">接続テスト中...</div>');
        
        $.ajax({
            url: '/seitekisitebilder/api/test-ftp.php',
            method: 'POST',
            data: $('#ftpSettingsForm').serialize(),
            success: function(response) {
                if (response.success) {
                    $('#ftpStatus').html('<div class="alert alert-success">接続成功！FTPサーバーに正常に接続できました。</div>');
                } else {
                    $('#ftpStatus').html(`<div class="alert alert-danger">接続失敗: ${response.error}</div>`);
                }
            },
            error: function(xhr) {
                let errorMsg = 'テスト中にエラーが発生しました。';
                if (xhr.responseJSON && xhr.responseJSON.error) {
                    errorMsg = xhr.responseJSON.error;
                }
                $('#ftpStatus').html(`<div class="alert alert-danger">${errorMsg}</div>`);
            },
            complete: function() {
                btn.prop('disabled', false).html(originalText);
            }
        });
    });
    
    // 設定保存
    $('#ftpSettingsForm').submit(function(e) {
        e.preventDefault();
        
        const profileName = $('#profileName').val();
        if (!profileName) {
            alert('設定名称を入力してください');
            return;
        }
        
        $('#ftpStatus').html('<div class="alert alert-info">保存中...</div>');
        
        $.ajax({
            url: '/seitekisitebilder/api/save-ftp-profile.php',
            method: 'POST',
            data: $(this).serialize(),
            success: function(response) {
                if (response.success) {
                    const profileName = $('#profileName').val();
                    const isNew = !$('#profileId').val();
                    const message = isNew ? 
                        `<div class="alert alert-success"><i class="bi bi-check-circle"></i> 「${profileName}」を追加しました。リストに表示されています。</div>` :
                        `<div class="alert alert-success"><i class="bi bi-check-circle"></i> 「${profileName}」を更新しました。</div>`;
                    
                    $('#ftpStatus').html(message);
                    
                    // フォームをリセット
                    if (isNew) {
                        $('#ftpSettingsForm')[0].reset();
                        $('#profileId').val('');
                        $('#cancelEdit').hide();
                    }
                    
                    // 1秒後にページ更新
                    setTimeout(() => {
                        // 新規追加の場合はセッションにメッセージを保存
                        if (isNew && response.new_id) {
                            sessionStorage.setItem('ftp_new_id', response.new_id);
                            sessionStorage.setItem('ftp_new_name', profileName);
                        }
                        location.reload();
                    }, 1000);
                } else {
                    $('#ftpStatus').html(`<div class="alert alert-danger">保存エラー: ${response.error}</div>`);
                }
            },
            error: function() {
                $('#ftpStatus').html('<div class="alert alert-danger">保存中にエラーが発生しました。</div>');
            }
        });
    });
});
</script>

<style>
.spin {
    display: inline-block;
    animation: spin 1s linear infinite;
}
@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}
@keyframes highlight {
    0% { background-color: #fff3cd; }
    100% { background-color: transparent; }
}
.highlight-new {
    animation: highlight 2s ease-out;
}
/* ボタンのスタイル */
#testConnection,
button[type="submit"],
#cancelEdit {
    white-space: nowrap;
    width: auto !important;
    flex: 0 0 auto;
}
/* インラインフレックスの調整 */
.d-inline-flex {
    display: inline-flex !important;
}
/* モバイルでのボタン配置 */
@media (max-width: 576px) {
    .d-inline-flex.gap-2 {
        display: flex !important;
        flex-wrap: wrap;
    }
    .d-inline-flex.gap-2 button {
        flex: 1 1 auto;
        min-width: 100px;
        max-width: 200px;
    }
}
</style>