<?php
// プリセットの読み込み
$presetsFile = __DIR__ . '/../../data/presets.json';
$presets = [];
if (file_exists($presetsFile)) {
    $presets = json_decode(file_get_contents($presetsFile), true) ?: [];
}
?>

<style>
/* Import refined theme */
@import url('../assets/css/nyx-refined.css');

/* Generation Page Specific */
.generation-container {
    max-width: 1400px;
    margin: 0 auto;
}

.preset-card {
    background: var(--nyx-bg-elevated);
    border: 1px solid var(--nyx-border);
    border-radius: 12px;
    padding: 1rem;
    cursor: pointer;
    transition: all 0.2s ease;
}

.preset-card:hover {
    border-color: var(--nyx-primary);
    transform: translateY(-2px);
    box-shadow: var(--nyx-shadow-md);
}

.preset-card.active {
    border-color: var(--nyx-primary);
    background: rgba(124, 58, 237, 0.1);
}

.model-card {
    padding: 1rem;
    border: 1px solid var(--nyx-border);
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s ease;
    position: relative;
    overflow: hidden;
}

.model-card.selected {
    border-color: var(--nyx-primary);
    background: rgba(124, 58, 237, 0.05);
}

.model-card .badge {
    position: absolute;
    top: 0.5rem;
    right: 0.5rem;
    font-size: 0.7rem;
}

.feature-toggle {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 1rem;
    background: rgba(255, 255, 255, 0.02);
    border-radius: 8px;
    margin-bottom: 1rem;
}

.toggle-switch {
    position: relative;
    width: 48px;
    height: 24px;
}

.toggle-switch input {
    opacity: 0;
    width: 0;
    height: 0;
}

.toggle-slider {
    position: absolute;
    cursor: pointer;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: rgba(255, 255, 255, 0.1);
    transition: .4s;
    border-radius: 24px;
}

.toggle-slider:before {
    position: absolute;
    content: "";
    height: 16px;
    width: 16px;
    left: 4px;
    bottom: 4px;
    background-color: white;
    transition: .4s;
    border-radius: 50%;
}

input:checked + .toggle-slider {
    background-color: var(--nyx-primary);
}

input:checked + .toggle-slider:before {
    transform: translateX(24px);
}

.generation-preview {
    position: fixed;
    right: 2rem;
    bottom: 2rem;
    width: 300px;
    background: var(--nyx-bg-elevated);
    border: 1px solid var(--nyx-border);
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: var(--nyx-shadow-lg);
    display: none;
}

.ai-status {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: rgba(124, 58, 237, 0.1);
    border-radius: 20px;
    font-size: 0.875rem;
}

.ai-status.active {
    animation: pulse-subtle 2s infinite;
}
</style>

<div class="generation-container">
    <!-- Header with Status -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="nyx-title">
                <i class="bi bi-stars"></i> AI記事生成スタジオ
            </h1>
            <p class="nyx-page-subtitle">最先端のAIモデルで高品質なコンテンツを作成</p>
        </div>
        <div class="ai-status" id="aiStatus">
            <div class="nyx-spinner" style="width: 16px; height: 16px; display: none;"></div>
            <span>待機中</span>
        </div>
    </div>

    <!-- Quick Presets -->
    <div class="glass-card mb-4">
        <h3 class="mb-3">クイックプリセット</h3>
        <div class="row g-3">
            <?php 
            $popularPresets = array_slice($presets, 0, 4, true);
            foreach ($popularPresets as $key => $preset): 
            ?>
            <div class="col-md-3">
                <div class="preset-card" onclick="selectPreset('<?php echo $key; ?>')">
                    <h5 class="mb-1"><?php echo htmlspecialchars($preset['name']); ?></h5>
                    <small class="text-muted"><?php echo htmlspecialchars($preset['model'] ?? 'デフォルト'); ?></small>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <div class="row">
        <!-- Main Form -->
        <div class="col-lg-8">
            <form id="generateForm" class="glass-card">
                <div class="mb-4">
                    <label class="nyx-label">記事タイトル</label>
                    <input type="text" class="nyx-form-control" id="title" 
                           placeholder="AIが魅力的なタイトルに改善します" required>
                    <small class="text-muted">ヒント: シンプルなタイトルでもAIが最適化します</small>
                </div>

                <div class="mb-4">
                    <label class="nyx-label">キーワード</label>
                    <div class="input-group">
                        <input type="text" class="nyx-form-control" id="keywords" 
                               placeholder="AI, 人工知能, 機械学習">
                        <button type="button" class="nyx-btn nyx-btn-secondary" onclick="suggestKeywords()">
                            <i class="bi bi-lightbulb"></i> 提案
                        </button>
                    </div>
                </div>

                <!-- AI Model Selection -->
                <div class="mb-4">
                    <label class="nyx-label">AIモデル選択</label>
                    <div class="row g-3">
                        <?php 
                        $featuredModels = [
                            'claude-opus-4-20250514' => ['provider' => 'anthropic', 'badge' => '最新'],
                            'gpt-4o' => ['provider' => 'openai', 'badge' => '人気'],
                            'gemini-2.5-pro' => ['provider' => 'google', 'badge' => '高速']
                        ];
                        
                        foreach ($featuredModels as $modelKey => $info):
                            $model = $config['models'][$info['provider']][$modelKey] ?? null;
                            if ($model):
                        ?>
                        <div class="col-md-4">
                            <div class="model-card" onclick="selectModel('<?php echo $modelKey; ?>')">
                                <span class="badge bg-primary"><?php echo $info['badge']; ?></span>
                                <h6><?php echo htmlspecialchars($model['name']); ?></h6>
                                <small class="text-muted"><?php echo htmlspecialchars($model['description'] ?? ''); ?></small>
                            </div>
                        </div>
                        <?php 
                            endif;
                        endforeach; 
                        ?>
                    </div>
                    <select class="form-select d-none" id="model">
                        <?php foreach ($config['models'] as $provider => $models): ?>
                            <?php foreach ($models as $key => $model): ?>
                                <option value="<?php echo htmlspecialchars($key); ?>">
                                    <?php echo htmlspecialchars($model['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        <?php endforeach; ?>
                    </select>
                </div>

                <!-- Advanced Features -->
                <div class="mb-4">
                    <h4 class="mb-3">拡張機能</h4>
                    
                    <div class="feature-toggle">
                        <div>
                            <h6 class="mb-0">Perplexity連携</h6>
                            <small class="text-muted">最新情報を自動収集</small>
                        </div>
                        <label class="toggle-switch">
                            <input type="checkbox" id="enable_perplexity">
                            <span class="toggle-slider"></span>
                        </label>
                    </div>

                    <div class="feature-toggle">
                        <div>
                            <h6 class="mb-0">画像自動生成</h6>
                            <small class="text-muted">セクション毎に画像を生成</small>
                        </div>
                        <label class="toggle-switch">
                            <input type="checkbox" id="enable_image">
                            <span class="toggle-slider"></span>
                        </label>
                    </div>

                    <div id="imageSettings" style="display: none;" class="mt-3 p-3 bg-black bg-opacity-25 rounded">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">画像モデル</label>
                                <select class="nyx-form-control" id="image_model">
                                    <option value="dall-e-3">DALL-E 3</option>
                                    <option value="imagen-3">Imagen 3</option>
                                    <option value="flux-dev">FLUX.1 Dev</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">スタイル</label>
                                <select class="nyx-form-control" id="image_style">
                                    <option value="photorealistic">写真のようなリアル</option>
                                    <option value="artistic">アーティスティック</option>
                                    <option value="anime">アニメ風</option>
                                    <option value="3d-render">3Dレンダリング</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Fine Tuning -->
                <div class="mb-4">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label d-flex justify-content-between">
                                <span>創造性</span>
                                <span class="text-primary" id="tempValue">0.7</span>
                            </label>
                            <input type="range" class="form-range" id="temperature" 
                                   min="0" max="1" step="0.1" value="0.7"
                                   style="accent-color: var(--nyx-primary);">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">文字数目安</label>
                            <select class="nyx-form-control" id="length_preset">
                                <option value="1000">短め（〜1000文字）</option>
                                <option value="2000" selected>標準（〜2000文字）</option>
                                <option value="4000">長め（〜4000文字）</option>
                                <option value="8000">詳細（〜8000文字）</option>
                            </select>
                        </div>
                    </div>
                </div>

                <!-- Generate Button -->
                <div class="text-center">
                    <button type="submit" class="nyx-btn nyx-btn-primary" style="padding: 1rem 3rem;">
                        <i class="bi bi-magic"></i> 記事を生成
                    </button>
                </div>
            </form>
        </div>

        <!-- Side Panel -->
        <div class="col-lg-4">
            <!-- Tips -->
            <div class="glass-card mb-4">
                <h4 class="mb-3"><i class="bi bi-lightbulb"></i> プロのヒント</h4>
                <ul class="list-unstyled">
                    <li class="mb-2">
                        <i class="bi bi-check-circle text-success"></i>
                        タイトルは短くても大丈夫。AIが最適化します
                    </li>
                    <li class="mb-2">
                        <i class="bi bi-check-circle text-success"></i>
                        キーワードは3-5個が理想的
                    </li>
                    <li class="mb-2">
                        <i class="bi bi-check-circle text-success"></i>
                        Perplexityで最新トレンドを反映
                    </li>
                    <li>
                        <i class="bi bi-check-circle text-success"></i>
                        画像生成で視覚的訴求力UP
                    </li>
                </ul>
            </div>

            <!-- Recent Generations -->
            <div class="glass-card">
                <h4 class="mb-3"><i class="bi bi-clock-history"></i> 最近の生成</h4>
                <div id="recentList">
                    <p class="text-muted">まだ生成履歴がありません</p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Generation Preview (Floating) -->
<div class="generation-preview" id="generationPreview">
    <h5>生成プレビュー</h5>
    <div id="previewContent">
        <p class="text-muted">生成中...</p>
    </div>
</div>

<!-- Result Modal -->
<div class="modal fade" id="resultModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content" style="background: var(--nyx-bg-elevated); border: 1px solid var(--nyx-border);">
            <div class="modal-header">
                <h5 class="modal-title">生成結果</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div id="generatedContent"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="nyx-btn nyx-btn-secondary" onclick="copyContent()">
                    <i class="bi bi-clipboard"></i> コピー
                </button>
                <button type="button" class="nyx-btn nyx-btn-primary" onclick="saveArticle()">
                    <i class="bi bi-save"></i> 保存
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// Form handling
document.getElementById('generateForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    
    const status = document.getElementById('aiStatus');
    status.classList.add('active');
    status.querySelector('span').textContent = '生成中...';
    status.querySelector('.nyx-spinner').style.display = 'inline-block';
    
    const formData = {
        action: 'generate',
        title: document.getElementById('title').value,
        keywords: document.getElementById('keywords').value,
        model: document.getElementById('model').value,
        temperature: parseFloat(document.getElementById('temperature').value),
        max_tokens: parseInt(document.getElementById('length_preset').value),
        enable_perplexity: document.getElementById('enable_perplexity').checked,
        enable_image: document.getElementById('enable_image').checked,
        image_model: document.getElementById('image_model').value,
        image_style: document.getElementById('image_style').value,
        output_format: 'html'
    };
    
    try {
        const response = await fetch('../api/service.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(formData)
        });
        
        const result = await response.json();
        
        if (result.success) {
            showResult(result);
            updateRecentList();
        } else {
            alert('エラー: ' + result.error);
        }
    } catch (error) {
        alert('通信エラー: ' + error.message);
    } finally {
        status.classList.remove('active');
        status.querySelector('span').textContent = '待機中';
        status.querySelector('.nyx-spinner').style.display = 'none';
    }
});

// UI Functions
function selectPreset(key) {
    document.querySelectorAll('.preset-card').forEach(card => card.classList.remove('active'));
    event.currentTarget.classList.add('active');
    // Load preset data
}

function selectModel(modelKey) {
    document.querySelectorAll('.model-card').forEach(card => card.classList.remove('selected'));
    event.currentTarget.classList.add('selected');
    document.getElementById('model').value = modelKey;
}

function showResult(result) {
    document.getElementById('generatedContent').innerHTML = result.content;
    new bootstrap.Modal(document.getElementById('resultModal')).show();
}

// Toggle handlers
document.getElementById('enable_image').addEventListener('change', (e) => {
    document.getElementById('imageSettings').style.display = e.target.checked ? 'block' : 'none';
});

document.getElementById('temperature').addEventListener('input', (e) => {
    document.getElementById('tempValue').textContent = e.target.value;
});

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    // Set default model
    selectModel('claude-opus-4-20250514');
});
</script>