<?php
// 直接アクセスの防止（index.php経由でない場合）
if (!isset($_SESSION)) {
    header('Location: ../index.php?page=presets');
    exit;
}

// プリセットの読み込み
$presetsFile = __DIR__ . '/../../data/presets.json';
$presets = [];
if (file_exists($presetsFile)) {
    $presets = json_decode(file_get_contents($presetsFile), true) ?: [];
}

// AJAX処理は api/preset-service.php に移動しました
?>

<div class="page-header">
    <h1><i class="bi bi-bookmark"></i> プリセット管理</h1>
    <p class="text-muted">よく使う設定をプリセットとして保存・管理します</p>
</div>

<div class="mb-4">
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#presetModal">
        <i class="bi bi-plus-circle"></i> 新規プリセット
    </button>
</div>

<div class="row g-3">
    <?php foreach ($presets as $key => $preset): ?>
        <div class="col-md-6 col-lg-4">
            <div class="card preset-card h-100">
                <div class="card-body">
                    <h5 class="card-title"><?php echo htmlspecialchars($preset['name']); ?></h5>
                    
                    <div class="preset-info">
                        <p class="mb-1">
                            <strong>モデル:</strong> 
                            <?php 
                            $modelName = 'Unknown';
                            // Venice AI プリセットの特別処理
                            if (str_starts_with($preset['model'], 'venice-')) {
                                $actualModel = substr($preset['model'], 7); // 'venice-' を削除
                                if (isset($config['models']['venice'][$actualModel])) {
                                    $modelName = $config['models']['venice'][$actualModel]['name'];
                                } else {
                                    $modelName = $preset['model']; // フォールバック
                                }
                            } else {
                                foreach ($config['models'] as $provider => $models) {
                                    if (isset($models[$preset['model']])) {
                                        $modelName = $models[$preset['model']]['name'];
                                        break;
                                    }
                                }
                            }
                            echo htmlspecialchars($modelName);
                            ?>
                        </p>
                        <p class="mb-1">
                            <strong>Temperature:</strong> <?php echo $preset['temperature']; ?>
                        </p>
                        <p class="mb-1">
                            <strong>最大トークン:</strong> <?php echo number_format($preset['max_tokens']); ?>
                        </p>
                        <p class="mb-0">
                            <strong>画像生成:</strong> 
                            <?php if ($preset['enable_image'] ?? false): ?>
                                <span class="badge bg-success">有効</span>
                                <?php echo htmlspecialchars($preset['image_style'] ?? 'realistic'); ?>
                            <?php else: ?>
                                <span class="badge bg-secondary">無効</span>
                            <?php endif; ?>
                        </p>
                    </div>
                </div>
                <div class="card-footer">
                    <div class="btn-group btn-group-sm w-100">
                        <button class="btn btn-outline-primary" onclick="editPreset('<?php echo htmlspecialchars($key); ?>')">
                            <i class="bi bi-pencil"></i> 編集
                        </button>
                        <?php if (!in_array($key, ['default', 'blog', 'news', 'product', 'gemini_blog'])): ?>
                            <button class="btn btn-outline-danger" onclick="deletePreset('<?php echo htmlspecialchars($key); ?>')">
                                <i class="bi bi-trash"></i> 削除
                            </button>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    <?php endforeach; ?>
</div>

<!-- プリセット編集モーダル -->
<div class="modal fade" id="presetModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">プリセット編集</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="presetForm">
                <div class="modal-body">
                    <input type="hidden" id="presetKey" name="key">
                    
                    <div class="mb-3">
                        <label for="presetName" class="form-label">プリセット名</label>
                        <input type="text" class="form-control" id="presetName" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="presetModel" class="form-label">モデル</label>
                        <select class="form-select" id="presetModel" name="model">
                            <?php foreach ($config['models'] as $provider => $models): ?>
                                <optgroup label="<?php echo ucfirst($provider); ?>">
                                    <?php foreach ($models as $key => $model): ?>
                                        <option value="<?php echo htmlspecialchars($key); ?>">
                                            <?php echo htmlspecialchars($model['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </optgroup>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="presetTemperature" class="form-label">Temperature</label>
                                <input type="number" class="form-control" id="presetTemperature" 
                                       name="temperature" min="0" max="1" step="0.1" value="0.7">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="presetMaxTokens" class="form-label">最大トークン</label>
                                <input type="number" class="form-control" id="presetMaxTokens" 
                                       name="max_tokens" min="100" max="128000" value="2000">
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="presetSystemPrompt" class="form-label">システムプロンプト</label>
                        <textarea class="form-control" id="presetSystemPrompt" name="system_prompt" rows="3"></textarea>
                    </div>
                    
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="checkbox" id="presetEnableImage" name="enable_image">
                        <label class="form-check-label" for="presetEnableImage">
                            画像生成を有効にする
                        </label>
                    </div>
                    
                    <div id="presetImageSettings" style="display: none;">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="presetImageModel" class="form-label">画像生成モデル</label>
                                    <select class="form-select" id="presetImageModel" name="image_model">
                                        <?php foreach ($config['image_models'] as $key => $model): ?>
                                            <option value="<?php echo htmlspecialchars($key); ?>">
                                                <?php echo htmlspecialchars($model['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="presetImageStyle" class="form-label">画像スタイル</label>
                                    <select class="form-select" id="presetImageStyle" name="image_style">
                                        <?php foreach ($config['image_styles'] as $key => $style): ?>
                                            <option value="<?php echo htmlspecialchars($key); ?>">
                                                <?php echo htmlspecialchars($style); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="presetImagePrompt" class="form-label">画像生成プロンプト</label>
                            <textarea class="form-control" id="presetImagePrompt" name="image_prompt" rows="2"></textarea>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">キャンセル</button>
                    <button type="submit" class="btn btn-primary">保存</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// プリセットデータ
const presets = <?php echo json_encode($presets); ?>;

// 画像設定の表示切替
document.getElementById('presetEnableImage').addEventListener('change', function() {
    document.getElementById('presetImageSettings').style.display = this.checked ? 'block' : 'none';
});

// プリセット編集
function editPreset(key) {
    const preset = presets[key];
    if (!preset) return;
    
    document.getElementById('presetKey').value = key;
    document.getElementById('presetName').value = preset.name;
    document.getElementById('presetModel').value = preset.model;
    document.getElementById('presetTemperature').value = preset.temperature;
    document.getElementById('presetMaxTokens').value = preset.max_tokens;
    document.getElementById('presetSystemPrompt').value = preset.system_prompt || '';
    document.getElementById('presetEnableImage').checked = preset.enable_image || false;
    document.getElementById('presetImageModel').value = preset.image_model || 'gpt-image-1';
    document.getElementById('presetImageStyle').value = preset.image_style || 'realistic';
    document.getElementById('presetImagePrompt').value = preset.image_prompt || '';
    
    document.getElementById('presetImageSettings').style.display = preset.enable_image ? 'block' : 'none';
    
    new bootstrap.Modal(document.getElementById('presetModal')).show();
}

// プリセット削除
function deletePreset(key) {
    if (!confirm('このプリセットを削除してもよろしいですか？')) return;
    
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'delete');
    formData.append('key', key);
    
    fetch('../api/preset-service.php', {
        method: 'POST',
        body: formData,
        credentials: 'same-origin'  // Cookie/セッション情報を含める
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('ネットワークエラー: ' + response.status);
        }
        return response.json();
    })
    .then(result => {
        if (result.success) {
            showToast('プリセットを削除しました', 'success');
            setTimeout(() => location.reload(), 1500);
        } else {
            showToast('エラー: ' + result.error, 'danger');
        }
    })
    .catch(error => {
        alert('通信エラー: ' + error.message);
    });
}

// フォーム送信
document.getElementById('presetForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('ajax', '1');
    formData.append('action', 'save');
    
    // キーが空の場合は削除（新規作成時にサーバー側で生成される）
    const keyField = formData.get('key');
    if (keyField === '') {
        formData.delete('key');
    }
    
    fetch('../api/preset-service.php', {
        method: 'POST',
        body: formData,
        credentials: 'same-origin'  // Cookie/セッション情報を含める
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('ネットワークエラー: ' + response.status);
        }
        return response.json();
    })
    .then(result => {
        if (result.success) {
            bootstrap.Modal.getInstance(document.getElementById('presetModal')).hide();
            showToast('プリセットを保存しました', 'success');
            
            // プリセットデータを更新
            if (result.preset && result.key) {
                presets[result.key] = result.preset;
                
                // 別のページ（generate.phpやdmm.php）のプリセット選択リストも更新するため
                // LocalStorageに保存
                localStorage.setItem('newPresetKey', result.key);
                localStorage.setItem('newPresetData', JSON.stringify(result.preset));
            }
            
            setTimeout(() => location.reload(), 1500);
        } else {
            showToast('エラー: ' + result.error, 'danger');
        }
    })
    .catch(error => {
        alert('通信エラー: ' + error.message);
    });
});

// 新規プリセット作成時の初期化
document.querySelector('[data-bs-target="#presetModal"]').addEventListener('click', function() {
    // フォームをリセット
    document.getElementById('presetForm').reset();
    document.getElementById('presetKey').value = '';
    document.getElementById('presetImageSettings').style.display = 'none';
});
// トースト表示関数
function showToast(message, type = 'info') {
    const toastHtml = `
        <div class="toast align-items-center text-white bg-${type} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">${message}</div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    let container = document.querySelector('.toast-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
    }
    
    container.insertAdjacentHTML('beforeend', toastHtml);
    const toastEl = container.lastElementChild;
    const toast = new bootstrap.Toast(toastEl);
    toast.show();
    
    toastEl.addEventListener('hidden.bs.toast', () => {
        toastEl.remove();
    });
}
</script>