<?php
// セッションチェックは親ファイル（index.php）で行われているため、ここでは不要

// 記事データを読み込み - history.jsonとarticles.jsonの両方を確認
$historyFile = __DIR__ . '/../../data/history.json';
$articlesFile = __DIR__ . '/../../data/articles.json';
$articles = [];

// まずhistory.jsonから読み込み
if (file_exists($historyFile)) {
    $historyData = json_decode(file_get_contents($historyFile), true);
    if (is_array($historyData) && !empty($historyData)) {
        $articles = $historyData;
    }
}

// history.jsonが空の場合、articles.jsonも確認
if (empty($articles) && file_exists($articlesFile)) {
    $articlesData = json_decode(file_get_contents($articlesFile), true);
    if (is_array($articlesData) && !empty($articlesData)) {
        $articles = $articlesData;
    }
}

// カテゴリー一覧を取得
$categories = array_unique(array_column($articles, 'category'));

// 編集モードの場合、既存の設定を読み込み
$editMode = false;
$editConfigId = '';
$editConfig = null;
if (isset($_GET['edit'])) {
    $editConfigId = $_GET['edit'];
    $siteConfigFile = __DIR__ . '/../../data/site-configs.json';
    if (file_exists($siteConfigFile)) {
        $siteConfigs = json_decode(file_get_contents($siteConfigFile), true) ?: [];
        if (isset($siteConfigs[$editConfigId])) {
            $editMode = true;
            $editConfig = $siteConfigs[$editConfigId];
        }
    }
}
?>

<div class="container mt-4">
    <h2>静的サイト生成</h2>
    
    <?php if (empty($articles)): ?>
    <div class="alert alert-warning">
        <h4>記事がありません</h4>
        <p>まず記事を生成してから、サイト生成を行ってください。</p>
        <a href="?page=generate" class="btn btn-primary">記事を生成する</a>
    </div>
    <?php else: ?>
    
    <div class="card">
        <div class="card-body">
            <form id="siteGeneratorForm">
                <h4>サイト設定 <?php if ($editMode): ?><span class="badge bg-info">編集モード</span><?php endif; ?></h4>
                <!-- Auto-save indicator -->
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <div></div>
                    <div id="saveIndicator" class="text-muted small">
                        <i class="bi bi-info-circle"></i> 変更は自動的に保存されます
                    </div>
                </div>
                <?php if ($editMode): ?>
                <input type="hidden" name="config_id" value="<?php echo htmlspecialchars($editConfigId); ?>">
                <input type="hidden" id="currentConfigId" value="<?php echo htmlspecialchars($editConfigId); ?>">
                <?php endif; ?>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label>サイト名</label>
                        <input type="text" class="form-control" name="site_name" value="<?php echo htmlspecialchars($editConfig['display_name'] ?? 'マイブログ'); ?>" required>
                    </div>
                    <div class="col-md-6">
                        <label>キャッチフレーズ</label>
                        <input type="text" class="form-control" name="catchphrase" value="<?php echo htmlspecialchars($editConfig['hero_catchphrase'] ?? 'あなたの毎日に、新しい発見を'); ?>">
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label>ドメイン名</label>
                        <input type="url" class="form-control" name="site_url" value="<?php echo htmlspecialchars($editConfig['url'] ?? 'https://example.com'); ?>" required>
                        <small class="form-text text-muted">https://を含むドメイン名のみ（サブディレクトリは含めない）</small>
                    </div>
                    <div class="col-md-6">
                        <label>テンプレート</label>
                        <select class="form-control" name="template">
                            <option value="modern-blog-1col" <?php echo ($editConfig['template'] ?? '') === 'modern-blog-1col' ? 'selected' : ''; ?>>モダンブログ（1列表示）</option>
                            <option value="modern-blog" <?php echo ($editConfig['template'] ?? '') === 'modern-blog' ? 'selected' : ''; ?>>モダンブログ（2列表示）</option>
                        </select>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label>1ページあたりの記事数</label>
                        <input type="number" class="form-control" name="articles_per_page" min="1" max="50" value="<?php echo htmlspecialchars($editConfig['articles_per_page'] ?? '10'); ?>" required>
                        <small class="form-text text-muted">トップページに表示する記事数（デフォルト: 10）</small>
                    </div>
                    <div class="col-md-6">
                        <label>サイト説明</label>
                        <input type="text" class="form-control" name="site_description" value="<?php echo htmlspecialchars($editConfig['description'] ?? ''); ?>" placeholder="サイトの簡潔な説明">
                        <small class="form-text text-muted">SEO用のサイト説明文</small>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-12">
                        <label>配置場所</label>
                        <div class="form-check">
                            <input class="form-check-input" type="radio" name="deployment_type" id="deployment_root" value="root" <?php echo (!$editMode || ($editConfig['deployment_type'] ?? 'root') === 'root') ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="deployment_root">
                                ドメイン直下（例: https://example.com/）
                            </label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="radio" name="deployment_type" id="deployment_subdir" value="subdirectory" <?php echo ($editMode && ($editConfig['deployment_type'] ?? '') === 'subdirectory') ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="deployment_subdir">
                                サブディレクトリ（例: https://example.com/blog/）
                            </label>
                        </div>
                        <div class="mt-2" id="subdirectory-input" style="display: <?php echo ($editMode && ($editConfig['deployment_type'] ?? '') === 'subdirectory') ? 'block' : 'none'; ?>;">
                            <label>ディレクトリ名</label>
                            <input type="text" class="form-control" name="subdirectory" placeholder="blog" pattern="[a-zA-Z0-9\-_]+" title="英数字、ハイフン、アンダースコアのみ使用可能" value="<?php echo htmlspecialchars($editConfig['subdirectory'] ?? ''); ?>">
                            <small class="form-text text-muted">URLの一部となるディレクトリ名を入力してください（スラッシュは不要）</small>
                        </div>
                    </div>
                </div>
                
                <h4 class="mt-4">フリースペース設定</h4>
                <div class="row mb-3">
                    <div class="col-md-12">
                        <label>トップページ上部フリースペース</label>
                        <textarea class="form-control free-space" name="top_free_space" rows="4" 
                                  placeholder="HTMLコードを入力（広告、お知らせなど）"><?php echo htmlspecialchars($editConfig['top_free_space'] ?? ''); ?></textarea>
                        <small class="form-text text-muted">トップページのメインコンテンツ上部に表示されるHTMLコンテンツ</small>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-12">
                        <label>トップページ下部フリースペース</label>
                        <textarea class="form-control free-space" name="bottom_free_space" rows="4" 
                                  placeholder="HTMLコードを入力（広告、関連リンクなど）"><?php echo htmlspecialchars($editConfig['bottom_free_space'] ?? ''); ?></textarea>
                        <small class="form-text text-muted">トップページのメインコンテンツ下部に表示されるHTMLコンテンツ</small>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-12">
                        <label>記事ページフリースペース</label>
                        <textarea class="form-control free-space" name="article_free_space" rows="4" 
                                  placeholder="HTMLコードを入力（広告、関連記事など）"><?php echo htmlspecialchars($editConfig['article_free_space'] ?? ''); ?></textarea>
                        <small class="form-text text-muted">各記事ページの記事本文下部に表示されるHTMLコンテンツ</small>
                    </div>
                </div>
                
                <h4 class="mt-4">記事の選択</h4>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label>カテゴリーで絞り込み</label>
                        <select class="form-control" id="categoryFilter">
                            <option value="">すべて</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo htmlspecialchars($category); ?>">
                                    <?php echo htmlspecialchars($category); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label>記事数</label>
                        <p class="form-control-plaintext">
                            <span id="selectedCount">0</span> / <?php echo count($articles); ?> 記事を選択中
                        </p>
                    </div>
                </div>
                
                <div class="article-selection" style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; padding: 10px;">
                    <div class="form-check mb-2">
                        <input type="checkbox" class="form-check-input" id="selectAll">
                        <label class="form-check-label" for="selectAll">
                            <strong>すべて選択</strong>
                        </label>
                    </div>
                    <hr>
                    
                    <?php foreach (array_reverse($articles) as $index => $article): ?>
                    <div class="form-check article-item" data-category="<?php echo htmlspecialchars($article['category'] ?? ''); ?>">
                        <input type="checkbox" class="form-check-input article-checkbox" 
                               name="articles[]" value="<?php echo $index; ?>" 
                               id="article_<?php echo $index; ?>">
                        <label class="form-check-label" for="article_<?php echo $index; ?>">
                            <strong><?php echo htmlspecialchars($article['title']); ?></strong>
                            <small class="text-muted">
                                - <?php echo $article['category'] ?? 'その他'; ?>
                                (<?php echo date('Y/m/d', strtotime($article['created_at'])); ?>)
                            </small>
                        </label>
                    </div>
                    <?php endforeach; ?>
                </div>
                
                <div class="mt-4">
                    <button type="button" class="btn btn-secondary" onclick="saveBasicInfo()">
                        <i class="bi bi-save"></i> 基本情報を保存
                    </button>
                    <button type="submit" class="btn btn-primary btn-lg">
                        <i class="bi bi-globe"></i> サイトを生成
                    </button>
                    <button type="button" class="btn btn-outline-secondary" onclick="clearDraft()">
                        <i class="bi bi-trash"></i> 下書きをクリア
                    </button>
                    <div id="generateStatus" class="mt-3"></div>
                </div>
            </form>
        </div>
    </div>
    
    <div id="generatedSite" class="card mt-4" style="display: none;">
        <div class="card-body">
            <h4>生成完了！</h4>
            <p>サイトが正常に生成されました。</p>
            <div class="row">
                <div class="col-md-6">
                    <h5>生成されたファイル</h5>
                    <ul id="fileList"></ul>
                </div>
                <div class="col-md-6">
                    <h5>次のステップ</h5>
                    <ol>
                        <li>FTPクライアントを開く</li>
                        <li>生成されたフォルダの中身をアップロード</li>
                        <li>サイトを確認</li>
                    </ol>
                </div>
            </div>
            <div class="mt-3">
                <a href="#" id="downloadLink" class="btn btn-success" style="display:none;">
                    <i class="bi bi-download"></i> ZIPでダウンロード
                </a>
                <div class="dropdown d-inline-block" id="ftpDeployDropdown" style="display:none;">
                    <button class="btn btn-primary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                        <i class="bi bi-cloud-upload"></i> FTPで自動デプロイ
                    </button>
                    <ul class="dropdown-menu" id="ftpProfileList">
                        <!-- FTPプロファイルリストが動的に挿入される -->
                    </ul>
                </div>
                <button class="btn btn-secondary" onclick="location.reload()">
                    <i class="bi bi-arrow-clockwise"></i> 新しいサイトを生成
                </button>
            </div>
        </div>
    </div>
</div>

<style>
    .spin {
        display: inline-block;
        animation: spin 1s linear infinite;
    }
    @keyframes spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }
</style>

<script>
$(document).ready(function() {
    // 配置場所の選択に応じてサブディレクトリ入力欄を表示/非表示
    $('input[name="deployment_type"]').change(function() {
        if ($(this).val() === 'subdirectory') {
            $('#subdirectory-input').show();
            $('input[name="subdirectory"]').prop('required', true);
        } else {
            $('#subdirectory-input').hide();
            $('input[name="subdirectory"]').prop('required', false);
        }
    });
    
    // 全選択/解除
    $('#selectAll').change(function() {
        $('.article-checkbox:visible').prop('checked', $(this).prop('checked'));
        updateSelectedCount();
    });
    
    // カテゴリーフィルター
    $('#categoryFilter').change(function() {
        const category = $(this).val();
        if (category === '') {
            $('.article-item').show();
        } else {
            $('.article-item').hide();
            $(`.article-item[data-category="${category}"]`).show();
        }
        updateSelectedCount();
    });
    
    // 選択数を更新
    $('.article-checkbox').change(updateSelectedCount);
    
    function updateSelectedCount() {
        const count = $('.article-checkbox:checked').length;
        $('#selectedCount').text(count);
    }
    
    // フォーム送信
    $('#siteGeneratorForm').submit(function(e) {
        e.preventDefault();
        
        const selectedArticles = $('.article-checkbox:checked').length;
        if (selectedArticles === 0) {
            alert('記事を選択してください。');
            return;
        }
        
        $('#generateStatus').html('<div class="alert alert-info">生成中... <i class="bi bi-arrow-repeat spin"></i></div>');
        
        $.ajax({
            url: '../api/generate-site.php',
            method: 'POST',
            data: $(this).serialize(),
            success: function(response) {
                if (response.success) {
                    $('#generateStatus').html('<div class="alert alert-success">サイトが生成されました！</div>');
                    $('#generatedSite').show();
                    
                    // ファイルリストを表示
                    const fileList = $('#fileList');
                    fileList.empty();
                    response.files.forEach(file => {
                        fileList.append(`<li>${file}</li>`);
                    });
                    
                    // 生成されたサイトのディレクトリと設定IDを保存
                    window.generatedSiteDir = response.output_dir;
                    window.siteConfigId = response.config_id;
                    
                    // FTP設定があるか確認してドロップダウンを表示
                    loadFtpProfiles();
                } else {
                    $('#generateStatus').html(`<div class="alert alert-danger">エラー: ${response.error}</div>`);
                }
            },
            error: function() {
                $('#generateStatus').html('<div class="alert alert-danger">生成中にエラーが発生しました。</div>');
            }
        });
    });
    
    // 初期カウント
    updateSelectedCount();
    
    // FTPプロファイルを読み込み
    function loadFtpProfiles() {
        $.ajax({
            url: '../api/get-ftp-profiles.php',
            method: 'GET',
            success: function(response) {
                if (response.success && response.profiles && Object.keys(response.profiles).length > 0) {
                    const profileList = $('#ftpProfileList');
                    profileList.empty();
                    
                    // プロファイルをドロップダウンに追加
                    Object.entries(response.profiles).forEach(([id, profile]) => {
                        profileList.append(`
                            <li>
                                <a class="dropdown-item" href="#" onclick="deployToFtp('${id}')">
                                    <i class="bi bi-hdd-network"></i> ${profile.name}
                                    <small class="text-muted d-block">${profile.host}</small>
                                </a>
                            </li>
                        `);
                    });
                    
                    // 設定追加リンク
                    profileList.append(`
                        <li><hr class="dropdown-divider"></li>
                        <li>
                            <a class="dropdown-item" href="?page=ftp-settings">
                                <i class="bi bi-plus-circle"></i> FTP設定を追加
                            </a>
                        </li>
                    `);
                    
                    $('#ftpDeployDropdown').show();
                } else {
                    // FTP設定がない場合
                    $('#ftpDeployDropdown').after(' <small class="text-muted">（<a href="?page=ftp-settings">FTP設定</a>が必要です）</small>');
                }
            }
        });
    }
    
    // FTPデプロイ処理
    window.deployToFtp = function(profileId) {
        if (!window.generatedSiteDir) {
            alert('サイトディレクトリが見つかりません');
            return;
        }
        
        if (!confirm('選択したFTP設定でサイトをデプロイしますか？')) {
            return;
        }
        
        $('#generateStatus').html('<div class="alert alert-info">FTPアップロード中... <i class="bi bi-arrow-repeat spin"></i></div>');
        
        $.ajax({
            url: '../api/deploy-to-ftp-profile.php',
            method: 'POST',
            data: { 
                site_dir: window.generatedSiteDir,
                profile_id: profileId,
                config_id: window.siteConfigId
            },
            success: function(response) {
                if (response.success) {
                    let message = '<div class="alert alert-success">';
                    message += '<h5>デプロイ完了！</h5>';
                    message += `<p>アップロード: ${response.results.uploaded.length} ファイル</p>`;
                    if (response.results.failed.length > 0) {
                        message += `<p class="text-warning">失敗: ${response.results.failed.length} ファイル</p>`;
                    }
                    message += '</div>';
                    $('#generateStatus').html(message);
                } else {
                    $('#generateStatus').html(`<div class="alert alert-danger">デプロイエラー: ${response.error}</div>`);
                }
            },
            error: function() {
                $('#generateStatus').html('<div class="alert alert-danger">デプロイ中にエラーが発生しました。</div>');
            }
        });
    };
});
</script>

<!-- Enhanced save functionality -->
<script src="../assets/js/site-generator-enhanced.js?v=<?php echo time(); ?>"></script>

<?php endif; ?>