<?php
// WordPress設定の読み込み
$wpConfigFile = __DIR__ . '/../../data/wordpress.json';
$wpConfig = [];
if (file_exists($wpConfigFile)) {
    $wpConfig = json_decode(file_get_contents($wpConfigFile), true) ?: [];
}

// 設定保存処理
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_wordpress') {
    $wpConfig = [
        'enabled' => isset($_POST['enabled']),
        'site_url' => $_POST['site_url'] ?? '',
        'username' => $_POST['username'] ?? '',
        'app_password' => $_POST['app_password'] ?? '',
        'default_status' => $_POST['default_status'] ?? 'draft',
        'default_author' => $_POST['default_author'] ?? '',
        'default_category' => $_POST['default_category'] ?? '',
        'auto_featured_image' => isset($_POST['auto_featured_image'])
    ];
    
    // dataディレクトリの確認
    $dataDir = dirname($wpConfigFile);
    if (!is_dir($dataDir)) {
        mkdir($dataDir, 0755, true);
    }
    
    if (file_put_contents($wpConfigFile, json_encode($wpConfig, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
        $success = 'WordPress設定を保存しました。';
    } else {
        $error = '設定の保存に失敗しました。';
    }
}

// 接続テスト
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'test_connection') {
    $testResult = testWordPressConnection($wpConfig);
}

function testWordPressConnection($config) {
    if (empty($config['site_url']) || empty($config['username']) || empty($config['app_password'])) {
        return ['success' => false, 'message' => '接続情報を入力してください。'];
    }
    
    $url = rtrim($config['site_url'], '/') . '/wp-json/wp/v2/users/me';
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Basic ' . base64_encode($config['username'] . ':' . $config['app_password'])
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200) {
        $user = json_decode($response, true);
        return [
            'success' => true,
            'message' => '接続成功！ユーザー: ' . ($user['name'] ?? 'Unknown')
        ];
    } else {
        return [
            'success' => false,
            'message' => '接続失敗。HTTPステータス: ' . $httpCode
        ];
    }
}
?>

<div class="page-header">
    <h1><i class="bi bi-wordpress"></i> WordPress連携設定</h1>
    <p class="text-muted">WordPressサイトへの自動投稿設定</p>
</div>

<?php if (isset($success)): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class="bi bi-check-circle"></i> <?php echo htmlspecialchars($success); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class="bi bi-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if (isset($testResult)): ?>
    <div class="alert alert-<?php echo $testResult['success'] ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
        <i class="bi bi-<?php echo $testResult['success'] ? 'check-circle' : 'x-circle'; ?>"></i> 
        <?php echo htmlspecialchars($testResult['message']); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-body">
                <form method="POST">
                    <input type="hidden" name="action" value="save_wordpress">
                    
                    <div class="mb-4">
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" id="enabled" name="enabled" 
                                   <?php echo ($wpConfig['enabled'] ?? false) ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="enabled">
                                WordPress連携を有効にする
                            </label>
                        </div>
                    </div>
                    
                    <fieldset id="wpSettings" <?php echo ($wpConfig['enabled'] ?? false) ? '' : 'disabled'; ?>>
                        <h5 class="mb-3">接続設定</h5>
                        
                        <div class="mb-3">
                            <label for="site_url" class="form-label">WordPressサイトURL</label>
                            <input type="url" class="form-control" id="site_url" name="site_url" 
                                   value="<?php echo htmlspecialchars($wpConfig['site_url'] ?? ''); ?>"
                                   placeholder="https://example.com">
                            <small class="text-muted">WordPressサイトのURL（末尾のスラッシュは不要）</small>
                        </div>
                        
                        <div class="mb-3">
                            <label for="username" class="form-label">ユーザー名</label>
                            <input type="text" class="form-control" id="username" name="username" 
                                   value="<?php echo htmlspecialchars($wpConfig['username'] ?? ''); ?>">
                            <small class="text-muted">WordPressのユーザー名</small>
                        </div>
                        
                        <div class="mb-3">
                            <label for="app_password" class="form-label">アプリケーションパスワード</label>
                            <input type="password" class="form-control" id="app_password" name="app_password" 
                                   value="<?php echo htmlspecialchars($wpConfig['app_password'] ?? ''); ?>">
                            <small class="text-muted">
                                <a href="#" data-bs-toggle="modal" data-bs-target="#appPasswordModal">
                                    アプリケーションパスワードの作成方法
                                </a>
                            </small>
                        </div>
                        
                        <hr class="my-4">
                        
                        <h5 class="mb-3">投稿設定</h5>
                        
                        <div class="mb-3">
                            <label for="default_status" class="form-label">デフォルトの投稿ステータス</label>
                            <select class="form-select" id="default_status" name="default_status">
                                <option value="draft" <?php echo ($wpConfig['default_status'] ?? '') === 'draft' ? 'selected' : ''; ?>>下書き</option>
                                <option value="publish" <?php echo ($wpConfig['default_status'] ?? '') === 'publish' ? 'selected' : ''; ?>>公開</option>
                                <option value="private" <?php echo ($wpConfig['default_status'] ?? '') === 'private' ? 'selected' : ''; ?>>非公開</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label for="default_author" class="form-label">デフォルトの投稿者ID（オプション）</label>
                            <input type="number" class="form-control" id="default_author" name="default_author" 
                                   value="<?php echo htmlspecialchars($wpConfig['default_author'] ?? ''); ?>">
                            <small class="text-muted">空欄の場合は認証ユーザーが投稿者になります</small>
                        </div>
                        
                        <div class="mb-3">
                            <label for="default_category" class="form-label">デフォルトのカテゴリーID（オプション）</label>
                            <input type="number" class="form-control" id="default_category" name="default_category" 
                                   value="<?php echo htmlspecialchars($wpConfig['default_category'] ?? ''); ?>">
                        </div>
                        
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="auto_featured_image" name="auto_featured_image" 
                                       <?php echo ($wpConfig['auto_featured_image'] ?? false) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="auto_featured_image">
                                    最初の画像を自動的にアイキャッチ画像に設定
                                </label>
                            </div>
                        </div>
                    </fieldset>
                    
                    <div class="d-flex justify-content-between">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> 設定を保存
                        </button>
                        <button type="button" class="btn btn-secondary" onclick="testConnection()">
                            <i class="bi bi-wifi"></i> 接続テスト
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-info-circle"></i> 設定ガイド</h5>
            </div>
            <div class="card-body">
                <h6>必要な要件</h6>
                <ul class="small">
                    <li>WordPress 5.6以降</li>
                    <li>REST APIが有効</li>
                    <li>アプリケーションパスワード機能</li>
                    <li>編集者以上の権限</li>
                </ul>
                
                <h6 class="mt-3">セットアップ手順</h6>
                <ol class="small">
                    <li>WordPressの管理画面にログイン</li>
                    <li>「ユーザー」→「プロフィール」を開く</li>
                    <li>「アプリケーションパスワード」セクションで新規作成</li>
                    <li>生成されたパスワードをコピー</li>
                    <li>このページに貼り付けて保存</li>
                </ol>
                
                <h6 class="mt-3">トラブルシューティング</h6>
                <ul class="small">
                    <li><strong>401エラー:</strong> 認証情報を確認</li>
                    <li><strong>404エラー:</strong> URLまたはREST APIを確認</li>
                    <li><strong>403エラー:</strong> 権限不足の可能性</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- アプリケーションパスワード説明モーダル -->
<div class="modal fade" id="appPasswordModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">アプリケーションパスワードの作成方法</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <h6>1. WordPressにログイン</h6>
                <p>WordPressの管理画面にログインします。</p>
                
                <h6>2. プロフィール画面を開く</h6>
                <p>「ユーザー」→「プロフィール」または右上のユーザー名をクリック。</p>
                
                <h6>3. アプリケーションパスワードセクション</h6>
                <p>ページ下部の「アプリケーションパスワード」セクションを探します。</p>
                
                <h6>4. 新しいパスワードを作成</h6>
                <p>アプリケーション名（例：AI Article Generator）を入力して「新しいアプリケーションパスワードを追加」をクリック。</p>
                
                <h6>5. パスワードをコピー</h6>
                <p>生成されたパスワード（スペース付き）をそのままコピーして、この設定画面に貼り付けます。</p>
                
                <div class="alert alert-warning mt-3">
                    <i class="bi bi-exclamation-triangle"></i> 
                    パスワードは一度しか表示されません。必ずコピーして安全な場所に保管してください。
                </div>
            </div>
        </div>
    </div>
</div>

<form method="POST" id="testForm" style="display: none;">
    <input type="hidden" name="action" value="test_connection">
</form>

<script>
// 有効/無効の切り替え
document.getElementById('enabled').addEventListener('change', function() {
    document.getElementById('wpSettings').disabled = !this.checked;
});

// 接続テスト
function testConnection() {
    document.getElementById('testForm').submit();
}
</script>