/**
 * カテゴリ選択の改善
 * 多数のカテゴリに対応するため、検索可能な選択方式に変更
 */
document.addEventListener('DOMContentLoaded', function() {
    
    // カテゴリ選択を改善する関数
    function enhanceCategorySelects() {
        // すべてのカテゴリ選択要素を取得
        const categorySelects = document.querySelectorAll('select[name="category"], select#categoryFilter');
        
        categorySelects.forEach(select => {
            // 既に処理済みの場合はスキップ
            if (select.dataset.enhanced === 'true') return;
            
            // datalistを使用した検索可能な入力フィールドに変換
            const selectId = select.id || 'category_' + Math.random().toString(36).substr(2, 9);
            const datalistId = 'datalist_' + selectId;
            
            // 現在の値を保持
            const currentValue = select.value;
            
            // datalist要素を作成
            const datalist = document.createElement('datalist');
            datalist.id = datalistId;
            
            // select要素のoptionをdatalistに移動
            const options = Array.from(select.options);
            const optionValues = [];
            
            options.forEach(option => {
                if (option.value) {
                    const datalistOption = document.createElement('option');
                    datalistOption.value = option.text || option.value;
                    datalistOption.setAttribute('data-value', option.value);
                    datalist.appendChild(datalistOption);
                    optionValues.push(option.text || option.value);
                }
            });
            
            // input要素を作成
            const input = document.createElement('input');
            input.type = 'text';
            input.name = select.name;
            input.id = selectId;
            input.className = select.className;
            input.placeholder = 'カテゴリを入力または選択...';
            input.setAttribute('list', datalistId);
            input.value = currentValue;
            
            // スタイルを追加
            input.style.width = '100%';
            
            // カスタムドロップダウンボタンを追加
            const wrapper = document.createElement('div');
            wrapper.style.position = 'relative';
            wrapper.style.display = 'inline-block';
            wrapper.style.width = '100%';
            
            const dropdownBtn = document.createElement('button');
            dropdownBtn.type = 'button';
            dropdownBtn.innerHTML = '▼';
            dropdownBtn.style.position = 'absolute';
            dropdownBtn.style.right = '5px';
            dropdownBtn.style.top = '50%';
            dropdownBtn.style.transform = 'translateY(-50%)';
            dropdownBtn.style.border = 'none';
            dropdownBtn.style.background = 'transparent';
            dropdownBtn.style.cursor = 'pointer';
            dropdownBtn.style.padding = '5px';
            dropdownBtn.style.fontSize = '12px';
            
            // カテゴリリストを表示するポップアップ
            const categoryList = document.createElement('div');
            categoryList.style.position = 'absolute';
            categoryList.style.top = '100%';
            categoryList.style.left = '0';
            categoryList.style.right = '0';
            categoryList.style.maxHeight = '400px';
            categoryList.style.overflowY = 'auto';
            categoryList.style.background = 'white';
            categoryList.style.border = '1px solid #ccc';
            categoryList.style.borderRadius = '4px';
            categoryList.style.boxShadow = '0 2px 4px rgba(0,0,0,0.1)';
            categoryList.style.display = 'none';
            categoryList.style.zIndex = '1000';
            
            // 検索フィールド
            const searchInput = document.createElement('input');
            searchInput.type = 'text';
            searchInput.placeholder = 'カテゴリを検索...';
            searchInput.style.width = '100%';
            searchInput.style.padding = '8px';
            searchInput.style.border = 'none';
            searchInput.style.borderBottom = '1px solid #eee';
            searchInput.style.boxSizing = 'border-box';
            
            categoryList.appendChild(searchInput);
            
            // カテゴリアイテムを追加
            const itemsContainer = document.createElement('div');
            optionValues.forEach(value => {
                const item = document.createElement('div');
                item.textContent = value;
                item.style.padding = '8px 12px';
                item.style.cursor = 'pointer';
                item.style.transition = 'background-color 0.2s';
                
                item.addEventListener('mouseover', () => {
                    item.style.backgroundColor = '#f5f5f5';
                });
                
                item.addEventListener('mouseout', () => {
                    item.style.backgroundColor = 'transparent';
                });
                
                item.addEventListener('click', () => {
                    input.value = value;
                    categoryList.style.display = 'none';
                    input.dispatchEvent(new Event('change'));
                });
                
                itemsContainer.appendChild(item);
            });
            
            categoryList.appendChild(itemsContainer);
            
            // 検索機能
            searchInput.addEventListener('input', (e) => {
                const searchTerm = e.target.value.toLowerCase();
                const items = itemsContainer.querySelectorAll('div');
                
                items.forEach(item => {
                    if (item.textContent.toLowerCase().includes(searchTerm)) {
                        item.style.display = 'block';
                    } else {
                        item.style.display = 'none';
                    }
                });
            });
            
            // ドロップダウンボタンのクリックイベント
            dropdownBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                const isVisible = categoryList.style.display === 'block';
                categoryList.style.display = isVisible ? 'none' : 'block';
                if (!isVisible) {
                    searchInput.value = '';
                    searchInput.dispatchEvent(new Event('input'));
                    searchInput.focus();
                }
            });
            
            // 外部クリックで閉じる
            document.addEventListener('click', (e) => {
                if (!wrapper.contains(e.target)) {
                    categoryList.style.display = 'none';
                }
            });
            
            // 要素を置き換え
            wrapper.appendChild(input);
            wrapper.appendChild(dropdownBtn);
            wrapper.appendChild(categoryList);
            wrapper.appendChild(datalist);
            
            select.parentNode.replaceChild(wrapper, select);
            
            // 処理済みフラグを設定
            input.dataset.enhanced = 'true';
        });
    }
    
    // 初回実行
    enhanceCategorySelects();
    
    // 動的に追加される要素にも対応
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList') {
                setTimeout(enhanceCategorySelects, 100);
            }
        });
    });
    
    // body要素を監視
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
    
    // モーダルが開いた時にも実行
    $(document).on('shown.bs.modal', function() {
        enhanceCategorySelects();
    });
});