// 新しいタブで生成する機能を追加
function generateInNewTab() {
    const form = document.getElementById('generateForm');
    
    // フォームのバリデーションチェック
    if (!form.checkValidity()) {
        form.reportValidity();
        return;
    }
    
    // 現在のフォームデータを収集
    const formData = {
        title: document.getElementById('title').value,
        keywords: document.getElementById('keywords').value || '',
        model: document.getElementById('model').value,
        user_prompt: document.getElementById('user_prompt').value,
        temperature: document.getElementById('temperature')?.value || 0.7,
        max_tokens: document.getElementById('max_tokens')?.value || 2000
    };
    
    // ユニークなIDを生成（タブ間で混同しないように）
    const generationId = 'gen_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
    
    // LocalStorageにデータを保存
    localStorage.setItem('pendingGeneration_' + generationId, JSON.stringify({
        data: formData,
        timestamp: Date.now()
    }));
    
    // 新しいウィンドウでgenerate.phpを開く（IDをパラメータとして渡す）
    window.open('?page=generate&auto_gen=' + generationId, '_blank');
}

// ページ読み込み時に自動生成をチェック
document.addEventListener('DOMContentLoaded', function() {
    // URLパラメータをチェック
    const urlParams = new URLSearchParams(window.location.search);
    const autoGenId = urlParams.get('auto_gen');
    
    // auto_genパラメータがある場合のみ処理
    if (autoGenId) {
        const pendingData = localStorage.getItem('pendingGeneration_' + autoGenId);
        
        if (pendingData) {
            try {
                const { data, timestamp } = JSON.parse(pendingData);
                
                // 10秒以内のデータのみ処理（古いデータは無視）
                if (Date.now() - timestamp < 10000) {
                    console.log('Auto-generating with ID:', autoGenId);
                    
                    // フォームに値を設定
                    if (document.getElementById('title')) {
                        document.getElementById('title').value = data.title;
                    }
                    if (document.getElementById('keywords')) {
                        document.getElementById('keywords').value = data.keywords;
                    }
                    if (document.getElementById('model')) {
                        document.getElementById('model').value = data.model;
                    }
                    if (document.getElementById('user_prompt')) {
                        document.getElementById('user_prompt').value = data.user_prompt;
                    }
                    if (document.getElementById('temperature')) {
                        document.getElementById('temperature').value = data.temperature;
                        const tempValue = document.getElementById('temperatureValue');
                        if (tempValue) {
                            tempValue.textContent = data.temperature;
                        }
                    }
                    if (document.getElementById('max_tokens')) {
                        document.getElementById('max_tokens').value = data.max_tokens;
                    }
                    
                    // LocalStorageをクリア（使用済みなので削除）
                    localStorage.removeItem('pendingGeneration_' + autoGenId);
                    
                    // URLからauto_genパラメータを削除（履歴を綺麗にする）
                    const newUrl = new URL(window.location);
                    newUrl.searchParams.delete('auto_gen');
                    window.history.replaceState({}, document.title, newUrl);
                    
                    // 少し待ってから自動的に生成開始
                    setTimeout(() => {
                        console.log('Starting auto-generation...');
                        const generateBtn = document.getElementById('generateBtn');
                        if (generateBtn) {
                            generateBtn.click();
                        }
                    }, 500);
                } else {
                    // 古いデータは削除
                    console.log('Removing expired generation data');
                    localStorage.removeItem('pendingGeneration_' + autoGenId);
                }
            } catch (e) {
                console.error('Error processing pending generation:', e);
                localStorage.removeItem('pendingGeneration_' + autoGenId);
            }
        }
    }
    
    // 古いLocalStorageデータをクリーンアップ（30秒以上前のもの）
    const now = Date.now();
    for (let i = 0; i < localStorage.length; i++) {
        const key = localStorage.key(i);
        if (key && key.startsWith('pendingGeneration_')) {
            try {
                const data = JSON.parse(localStorage.getItem(key));
                if (data && data.timestamp && (now - data.timestamp > 30000)) {
                    console.log('Cleaning up old generation data:', key);
                    localStorage.removeItem(key);
                }
            } catch (e) {
                // 無効なデータは削除
                localStorage.removeItem(key);
            }
        }
    }
});