// Modal Fix - Prevent freezing and backdrop issues

(function() {
    'use strict';
    
    // Track all active modals
    const activeModals = new Set();
    
    // Override Bootstrap modal show
    const originalShow = bootstrap.Modal.prototype.show;
    bootstrap.Modal.prototype.show = function() {
        const modalEl = this._element;
        activeModals.add(modalEl);
        
        // Clean up any stray backdrops before showing
        cleanupBackdrops();
        
        // Call original show
        originalShow.call(this);
    };
    
    // Override Bootstrap modal hide
    const originalHide = bootstrap.Modal.prototype.hide;
    bootstrap.Modal.prototype.hide = function() {
        const modalEl = this._element;
        
        // Call original hide
        originalHide.call(this);
        
        // Schedule cleanup after animation
        setTimeout(() => {
            activeModals.delete(modalEl);
            cleanupBackdrops();
            resetBodyState();
        }, 300);
    };
    
    // Clean up stray backdrops
    function cleanupBackdrops() {
        const backdrops = document.querySelectorAll('.modal-backdrop');
        const validBackdrops = Array.from(activeModals).filter(modal => {
            return modal.classList.contains('show');
        }).length;
        
        // Remove extra backdrops
        if (backdrops.length > validBackdrops) {
            backdrops.forEach((backdrop, index) => {
                if (index >= validBackdrops) {
                    backdrop.remove();
                }
            });
        }
    }
    
    // Reset body state when no modals are active
    function resetBodyState() {
        if (activeModals.size === 0) {
            document.body.classList.remove('modal-open');
            document.body.style.removeProperty('overflow');
            document.body.style.removeProperty('padding-right');
            document.body.style.removeProperty('position');
            
            // Final cleanup of any remaining backdrops
            document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        }
    }
    
    // Handle modal disposal
    const originalDispose = bootstrap.Modal.prototype.dispose;
    bootstrap.Modal.prototype.dispose = function() {
        const modalEl = this._element;
        activeModals.delete(modalEl);
        
        // Call original dispose
        originalDispose.call(this);
        
        // Clean up
        setTimeout(() => {
            cleanupBackdrops();
            resetBodyState();
        }, 300);
    };
    
    // Global cleanup function
    window.forceCleanupModals = function() {
        // Hide all active modals
        document.querySelectorAll('.modal.show').forEach(modal => {
            const bsModal = bootstrap.Modal.getInstance(modal);
            if (bsModal) {
                bsModal.hide();
            }
            modal.classList.remove('show');
            modal.style.display = 'none';
        });
        
        // Clear tracking
        activeModals.clear();
        
        // Force cleanup
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        resetBodyState();
    };
    
    // Auto-cleanup on page visibility change
    document.addEventListener('visibilitychange', function() {
        if (!document.hidden) {
            // Page is visible again, check for stuck modals
            setTimeout(() => {
                const hasVisibleModal = document.querySelector('.modal.show');
                const hasBackdrop = document.querySelector('.modal-backdrop');
                
                if (hasBackdrop && !hasVisibleModal) {
                    // Stuck backdrop without modal
                    forceCleanupModals();
                }
            }, 100);
        }
    });
    
    // Handle escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            const visibleModals = document.querySelectorAll('.modal.show');
            if (visibleModals.length > 0) {
                const lastModal = visibleModals[visibleModals.length - 1];
                const bsModal = bootstrap.Modal.getInstance(lastModal);
                if (bsModal && bsModal._config.keyboard !== false) {
                    bsModal.hide();
                }
            }
        }
    });
    
    // Periodic cleanup check (every 5 seconds)
    setInterval(() => {
        const hasBackdrop = document.querySelector('.modal-backdrop');
        const hasVisibleModal = document.querySelector('.modal.show');
        const hasModalOpen = document.body.classList.contains('modal-open');
        
        // Check for inconsistent state
        if (hasBackdrop && !hasVisibleModal) {
            console.warn('Modal cleanup: Found backdrop without visible modal');
            forceCleanupModals();
        } else if (hasModalOpen && !hasVisibleModal && !hasBackdrop) {
            console.warn('Modal cleanup: Body has modal-open class without modal');
            resetBodyState();
        }
    }, 5000);
    
})();