// Nyx AI Studio - Interactive Effects

document.addEventListener('DOMContentLoaded', function() {
    // Create particle container
    createParticleBackground();
    
    // Add hover effects to buttons
    addButtonEffects();
    
    // Initialize tooltips
    initializeTooltips();
    
    // Add card animations
    animateCards();
    
    // Create aurora background effect
    createAuroraEffect();
});

// Particle Background Effect
function createParticleBackground() {
    const particleContainer = document.createElement('div');
    particleContainer.className = 'particle-container';
    document.body.appendChild(particleContainer);
    
    // Create stars
    for (let i = 0; i < 50; i++) {
        createStar(particleContainer);
    }
    
    // Create floating particles
    setInterval(() => {
        if (document.querySelectorAll('.particle-star').length < 30) {
            createFloatingParticle(particleContainer);
        }
    }, 2000);
}

function createStar(container) {
    const star = document.createElement('div');
    star.className = 'star';
    star.style.left = Math.random() * 100 + '%';
    star.style.top = Math.random() * 100 + '%';
    star.style.animationDelay = Math.random() * 3 + 's';
    star.style.animationDuration = (Math.random() * 3 + 2) + 's';
    container.appendChild(star);
}

function createFloatingParticle(container) {
    const particle = document.createElement('div');
    particle.className = 'particle-star';
    particle.style.left = Math.random() * 100 + '%';
    particle.style.animationDelay = Math.random() * 5 + 's';
    particle.style.animationDuration = (Math.random() * 10 + 10) + 's';
    container.appendChild(particle);
    
    // Remove particle after animation
    setTimeout(() => {
        particle.remove();
    }, 20000);
}

// Button Effects
function addButtonEffects() {
    const buttons = document.querySelectorAll('.neon-button');
    
    buttons.forEach(button => {
        button.addEventListener('mouseenter', function(e) {
            createRipple(e, this);
        });
        
        button.addEventListener('click', function(e) {
            createClickEffect(e, this);
        });
    });
}

function createRipple(e, button) {
    const ripple = document.createElement('span');
    ripple.className = 'ripple';
    
    const rect = button.getBoundingClientRect();
    const size = Math.max(rect.width, rect.height);
    const x = e.clientX - rect.left - size / 2;
    const y = e.clientY - rect.top - size / 2;
    
    ripple.style.width = ripple.style.height = size + 'px';
    ripple.style.left = x + 'px';
    ripple.style.top = y + 'px';
    
    button.appendChild(ripple);
    
    setTimeout(() => {
        ripple.remove();
    }, 600);
}

function createClickEffect(e, button) {
    // Create multiple sparkles
    for (let i = 0; i < 5; i++) {
        createSparkle(e.clientX, e.clientY);
    }
}

function createSparkle(x, y) {
    const sparkle = document.createElement('div');
    sparkle.className = 'sparkle';
    sparkle.style.left = x + 'px';
    sparkle.style.top = y + 'px';
    
    const angle = Math.random() * Math.PI * 2;
    const velocity = Math.random() * 50 + 50;
    const tx = Math.cos(angle) * velocity;
    const ty = Math.sin(angle) * velocity;
    
    sparkle.style.setProperty('--tx', tx + 'px');
    sparkle.style.setProperty('--ty', ty + 'px');
    
    document.body.appendChild(sparkle);
    
    setTimeout(() => {
        sparkle.remove();
    }, 1000);
}

// Card Animations
function animateCards() {
    const cards = document.querySelectorAll('.glass-card');
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach((entry, index) => {
            if (entry.isIntersecting) {
                setTimeout(() => {
                    entry.target.classList.add('card-entrance');
                }, index * 100);
                observer.unobserve(entry.target);
            }
        });
    }, { threshold: 0.1 });
    
    cards.forEach(card => {
        observer.observe(card);
    });
}

// Aurora Background Effect
function createAuroraEffect() {
    const aurora = document.createElement('div');
    aurora.className = 'aurora-bg';
    document.body.appendChild(aurora);
}

// Initialize Tooltips
function initializeTooltips() {
    const elements = document.querySelectorAll('[data-tooltip]');
    
    elements.forEach(element => {
        element.addEventListener('mouseenter', function() {
            const tooltip = document.createElement('div');
            tooltip.className = 'nyx-tooltip';
            tooltip.textContent = this.getAttribute('data-tooltip');
            
            document.body.appendChild(tooltip);
            
            const rect = this.getBoundingClientRect();
            tooltip.style.left = rect.left + rect.width / 2 - tooltip.offsetWidth / 2 + 'px';
            tooltip.style.top = rect.top - tooltip.offsetHeight - 10 + 'px';
            
            this._tooltip = tooltip;
        });
        
        element.addEventListener('mouseleave', function() {
            if (this._tooltip) {
                this._tooltip.remove();
                delete this._tooltip;
            }
        });
    });
}

// Loading Animation Helper
function showLoading(element) {
    const loader = document.createElement('div');
    loader.className = 'nyx-loader';
    element.appendChild(loader);
    return loader;
}

function hideLoading(loader) {
    if (loader && loader.parentNode) {
        loader.remove();
    }
}

// Success Animation
function showSuccess(message) {
    const notification = document.createElement('div');
    notification.className = 'nyx-notification success-notification';
    notification.innerHTML = `
        <svg width="24" height="24" viewBox="0 0 24 24" fill="none">
            <path class="success-checkmark" d="M5 12l5 5L20 7" stroke="currentColor" stroke-width="2"/>
        </svg>
        <span>${message}</span>
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.classList.add('show');
    }, 10);
    
    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => {
            notification.remove();
        }, 300);
    }, 3000);
}

// Export functions for use in other scripts
window.NyxEffects = {
    showLoading,
    hideLoading,
    showSuccess,
    createRipple,
    createSparkle
};