// Enhanced Site Generator with Basic Info Save Functionality

// Auto-save functionality
let autoSaveTimer = null;
let currentConfigId = null;

// Initialize when document is ready
$(document).ready(function() {
    // Check if we're in edit mode
    const urlParams = new URLSearchParams(window.location.search);
    const editConfigId = urlParams.get('edit');
    if (editConfigId) {
        currentConfigId = editConfigId;
        console.log('Edit mode: ' + currentConfigId);
    }
    
    // Load saved draft if exists
    loadDraft();
    
    // Add save button to basic info section
    addSaveButton();
    
    // Monitor form changes for auto-save
    $('#siteGeneratorForm input, #siteGeneratorForm select, #siteGeneratorForm textarea').on('change input', function() {
        showSaveStatus('unsaved');
        
        // Clear existing timer
        if (autoSaveTimer) {
            clearTimeout(autoSaveTimer);
        }
        
        // Set new timer for auto-save (3 seconds after last change)
        autoSaveTimer = setTimeout(function() {
            saveBasicInfo(true); // true = auto-save
        }, 3000);
    });
    
    // Warn before leaving if unsaved changes
    window.addEventListener('beforeunload', function(e) {
        if ($('#saveStatus').hasClass('unsaved')) {
            e.preventDefault();
            e.returnValue = '保存されていない変更があります。このページを離れますか？';
        }
    });
});

// Add save button to the form
function addSaveButton() {
    const basicInfoSection = $('#siteGeneratorForm h4:first').parent();
    
    // Add save status indicator
    basicInfoSection.append(`
        <div id="saveStatus" class="float-end saved">
            <span class="badge bg-secondary">
                <i class="bi bi-check-circle"></i> 保存済み
            </span>
        </div>
    `);
    
    // Add save button next to generate button
    const generateButton = $('#siteGeneratorForm button[type="submit"]');
    generateButton.before(`
        <button type="button" class="btn btn-secondary me-2" onclick="saveBasicInfo()">
            <i class="bi bi-save"></i> 基本情報を保存
        </button>
    `);
    
    // Add CSS for save status
    $('<style>')
        .text(`
            #saveStatus.saved .badge { background-color: #28a745 !important; }
            #saveStatus.unsaved .badge { background-color: #ffc107 !important; }
            #saveStatus.saving .badge { background-color: #17a2b8 !important; }
            #saveStatus.error .badge { background-color: #dc3545 !important; }
        `)
        .appendTo('head');
}

// Show save status
function showSaveStatus(status) {
    const statusElement = $('#saveStatus');
    const badge = statusElement.find('.badge');
    
    statusElement.removeClass('saved unsaved saving error').addClass(status);
    
    switch(status) {
        case 'saved':
            badge.html('<i class="bi bi-check-circle"></i> 保存済み');
            break;
        case 'unsaved':
            badge.html('<i class="bi bi-exclamation-circle"></i> 未保存');
            break;
        case 'saving':
            badge.html('<i class="bi bi-arrow-repeat spin"></i> 保存中...');
            break;
        case 'error':
            badge.html('<i class="bi bi-x-circle"></i> エラー');
            break;
    }
}

// Save basic information
function saveBasicInfo(isAutoSave = false) {
    showSaveStatus('saving');
    
    // Collect basic info data
    const basicInfo = {
        config_id: currentConfigId,
        site_name: $('input[name="site_name"]').val(),
        catchphrase: $('input[name="catchphrase"]').val(),
        site_url: $('input[name="site_url"]').val(),
        template: $('select[name="template"]').val(),
        articles_per_page: $('input[name="articles_per_page"]').val(),
        site_description: $('input[name="site_description"]').val(),
        deployment_type: $('input[name="deployment_type"]:checked').val(),
        subdirectory: $('input[name="subdirectory"]').val(),
        top_free_space: $('textarea[name="top_free_space"]').val(),
        bottom_free_space: $('textarea[name="bottom_free_space"]').val(),
        article_free_space: $('textarea[name="article_free_space"]').val()
    };
    
    // Save to localStorage as backup
    localStorage.setItem('siteGeneratorDraft', JSON.stringify(basicInfo));
    
    // Save to server
    $.ajax({
        url: '../api/save-site-config.php',
        method: 'POST',
        contentType: 'application/json',
        data: JSON.stringify(basicInfo),
        success: function(response) {
            if (response.success) {
                showSaveStatus('saved');
                if (!currentConfigId) {
                    currentConfigId = response.config_id;
                }
                
                if (!isAutoSave) {
                    showToast('基本情報を保存しました', 'success');
                }
                
                console.log('Saved with config ID: ' + currentConfigId);
            } else {
                showSaveStatus('error');
                showToast('保存エラー: ' + response.error, 'danger');
            }
        },
        error: function() {
            showSaveStatus('error');
            showToast('保存中にエラーが発生しました', 'danger');
        }
    });
}

// Load draft from localStorage or server
function loadDraft() {
    // Try to load from localStorage first
    const draft = localStorage.getItem('siteGeneratorDraft');
    if (draft && !currentConfigId) { // Only use localStorage if not in edit mode
        try {
            const data = JSON.parse(draft);
            
            // Populate form fields
            if (data.site_name) $('input[name="site_name"]').val(data.site_name);
            if (data.catchphrase) $('input[name="catchphrase"]').val(data.catchphrase);
            if (data.site_url) $('input[name="site_url"]').val(data.site_url);
            if (data.template) $('select[name="template"]').val(data.template);
            if (data.articles_per_page) $('input[name="articles_per_page"]').val(data.articles_per_page);
            if (data.site_description) $('input[name="site_description"]').val(data.site_description);
            if (data.deployment_type) {
                $(`input[name="deployment_type"][value="${data.deployment_type}"]`).prop('checked', true).trigger('change');
            }
            if (data.subdirectory) $('input[name="subdirectory"]').val(data.subdirectory);
            if (data.top_free_space) $('textarea[name="top_free_space"]').val(data.top_free_space);
            if (data.bottom_free_space) $('textarea[name="bottom_free_space"]').val(data.bottom_free_space);
            if (data.article_free_space) $('textarea[name="article_free_space"]').val(data.article_free_space);
            
            console.log('Loaded draft from localStorage');
            showToast('下書きを読み込みました', 'info');
        } catch (e) {
            console.error('Failed to load draft:', e);
        }
    }
}

// Clear draft
function clearDraft() {
    if (confirm('保存された下書きをクリアしますか？')) {
        localStorage.removeItem('siteGeneratorDraft');
        currentConfigId = null;
        $('#siteGeneratorForm')[0].reset();
        showSaveStatus('saved');
        showToast('下書きをクリアしました', 'success');
    }
}

// Show toast notification
function showToast(message, type = 'info') {
    const toastHtml = `
        <div class="toast align-items-center text-white bg-${type} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">${message}</div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    let container = document.querySelector('.toast-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
    }
    
    container.insertAdjacentHTML('beforeend', toastHtml);
    const toastEl = container.lastElementChild;
    const toast = new bootstrap.Toast(toastEl);
    toast.show();
    
    toastEl.addEventListener('hidden.bs.toast', () => {
        toastEl.remove();
    });
}